package ru.yandex.webmaster3.storage.user.dao;

import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.google.common.base.Preconditions;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author iceflame
 */
@Repository
public class UserNotificationChannelsYDao extends AbstractYDao {

    public UserNotificationChannelsYDao() {
        super(PREFIX_USER, "user_notification_channels");
    }

    private static final ValueDataMapper<Pair<Long, Pair<NotificationType, Set<NotificationChannel>>>> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, Pair::getLeft),
            Pair.of(F.NOTIFICATION_TYPE, pair -> pair.getRight().getLeft()),
            Pair.of(F.CHANNELS, pair -> pair.getRight().getRight())
    );

    static {
        // на всякий случай
        Preconditions.checkState(NotificationType.values().length <= 1000, "Too many notification types!");
    }

    public void batchInsert(Collection<Pair<Long, Pair<NotificationType, Set<NotificationChannel>>>> batch) {
        batchInsert(INSERT_VALUE_MAPPER, batch).execute();
    }

    public void updateSettings(long userId, NotificationType notificationType, Set<NotificationChannel> channels) {
        upsert(
                F.USER_ID.value(userId),
                F.NOTIFICATION_TYPE.value(notificationType),
                F.CHANNELS.value(channels)
        ).execute();
    }

    public Map<NotificationType, Set<NotificationChannel>> getUserSettings(long userId) {
        return select(TYPE_AND_CHANNELS)
                .where(F.USER_ID.eq(userId))
                .queryForList()
                .stream().collect(W3Collectors.toEnumMap(NotificationType.class));
    }


    public Map<Long, Set<NotificationChannel>> getChannelsInfo(Collection<Long> userIds, NotificationType notificationType) {
        Preconditions.checkArgument(userIds.size() <= 1000);
        return select(USER_CHANNELS)
                .where(F.USER_ID.in(userIds))
                .and(F.NOTIFICATION_TYPE.eq(notificationType))
                .queryForList().stream().collect(W3Collectors.toHashMap());
    }

    public Map<Long, Map<NotificationType, Set<NotificationChannel>>> getChannelsInfo(Collection<Long> userIds) {
        // тут может быть гораздо больше тысячи записей, поэтому придется сортировать
        Map<Long, Map<NotificationType, Set<NotificationChannel>>> result = new HashMap<>();
        select(MAPPER)
                .where(F.USER_ID.in(userIds))
                .queryForList(
                        Pair.of(F.USER_ID, Pair::getLeft),
                        Pair.of(F.NOTIFICATION_TYPE, pair -> pair.getRight().getLeft())
                ).forEach(p ->
                result.computeIfAbsent(p.getLeft(), k -> new EnumMap<>(NotificationType.class)).put(p.getRight().getLeft(), p.getRight().getRight())
        );
        return result;
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private static final DataMapper<Pair<Long, Set<NotificationChannel>>> USER_CHANNELS = F.USER_ID.combine(F.CHANNELS, Pair::of);
    private static final DataMapper<Pair<NotificationType, Set<NotificationChannel>>> TYPE_AND_CHANNELS = DataMapper.create(F.NOTIFICATION_TYPE, F.CHANNELS, Pair::of);
    private static final DataMapper<Pair<Long, Pair<NotificationType, Set<NotificationChannel>>>> MAPPER = F.USER_ID.combine(TYPE_AND_CHANNELS, Pair::of);

    private interface F {
        TypeReference<Set<NotificationChannel>> NOTIFICATION_CHANNEL_SET_REFERENCE = new TypeReference<>() {
        };

        ObjectMapper OM = JsonMapping.OM.copy().enable(SerializationFeature.WRITE_ENUMS_USING_INDEX);

        Field<Long> USER_ID = Fields.longField("user_id");
        Field<NotificationType> NOTIFICATION_TYPE = Fields.intEnumField("type", NotificationType.R);
        Field<Set<NotificationChannel>> CHANNELS = Fields.jsonField2("channels", NOTIFICATION_CHANNEL_SET_REFERENCE, OM);
    }

}
