package ru.yandex.webmaster3.storage.user.dao;

import java.util.Collection;
import java.util.Map;
import java.util.function.Consumer;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.user.notification.UserSetEmailInfo;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author iceflame
 */
@Repository
public class UserNotificationEmailYDao extends AbstractYDao {

    public UserNotificationEmailYDao() {
        super(PREFIX_USER, "user_notification_email");
    }

    private static final ValueDataMapper<Pair<Long, UserSetEmailInfo>> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, Pair::getLeft),
            Pair.of(F.EMAIL, pair -> pair.getRight().getEmail()),
            Pair.of(F.VERIFIED, pair -> pair.getRight().isVerified())
    );

    public void batchInsert(Collection<Pair<Long, UserSetEmailInfo>> records) {
        batchInsert(INSERT_VALUE_MAPPER, records).execute();
    }

    public void updateVerifiedState(long userId, String email, boolean verified) {
        update()
                .with(F.VERIFIED.set(verified))
                .where(F.USER_ID.eq(userId))
                .and(F.EMAIL.eq(email))
                .execute();
    }

    public void saveEmail(long userId, String email) {
        upsert(
                F.USER_ID.value(userId),
                F.EMAIL.value(email),
                F.VERIFIED.value(true)
        ).execute();
    }

    public String getEmail(Long userId) {
        UserSetEmailInfo settings = getEmailSettings(userId);
        if (settings != null && settings.isVerified()) {
            return settings.getEmail();
        } else {
            return null;
        }
    }

    @Nullable
    public UserSetEmailInfo getEmailSettings(Long userId) {
        return select(MAPPER)
                .where(F.USER_ID.eq(userId))
                .queryOne();
    }

    public Map<Long, String> getUserEmails(Collection<Long> userIds) {
        return getUserEmailsSettings(userIds).entrySet().stream()
                .filter(e -> e.getValue().isVerified() && e.getValue().getEmail() != null)
                .map(e -> Pair.of(e.getKey(), e.getValue().getEmail()))
                .collect(W3Collectors.toHashMap());
    }

    public Map<Long, UserSetEmailInfo> getUserEmailsSettings(Collection<Long> userIds) {
        Preconditions.checkArgument(userIds.size() <= 1000, "Batch is too big");
        return select(WITH_USER_MAPPER).where(F.USER_ID.in(userIds)).queryForList().stream().collect(W3Collectors.toHashMap());
    }

    public void forEach(Consumer<Pair<Long, UserSetEmailInfo>> consumer) {
        streamReader(WITH_USER_MAPPER, consumer);
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private static final DataMapper<UserSetEmailInfo> MAPPER = DataMapper.create(F.EMAIL, F.VERIFIED, UserSetEmailInfo::new);
    private static final DataMapper<Pair<Long, UserSetEmailInfo>> WITH_USER_MAPPER = DataMapper.create(F.USER_ID, MAPPER, Pair::of);

    private interface F {
        Field<Long> USER_ID = Fields.longField("user_id");
        Field<String> EMAIL = Fields.stringField("email").makeOptional();
        Field<Boolean> VERIFIED = Fields.boolField("verified").withDefault(true);
    }
}
