package ru.yandex.webmaster3.storage.user.dao;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationSetting;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.FieldMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

/**
 * @author aherman
 */
@Slf4j
@Repository
public class UserNotificationHostSettingsYDao extends AbstractYDao {

    public UserNotificationHostSettingsYDao() {
        super(PREFIX_USER, "user_notification_host_settings");
    }

    private static final ValueDataMapper<Pair<Long, HostNotificationSetting>> INSERT_VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.USER_ID, Pair::getLeft),
            Pair.of(F.HOST_ID, pair -> pair.getRight().getHostId()),
            Pair.of(F.TYPE, pair -> pair.getRight().getType()),
            Pair.of(F.CHANNEL, pair -> pair.getRight().getChannel()),
            Pair.of(F.MODE, pair -> pair.getRight().getMode())
    );

    public void batchInsert(Collection<Pair<Long, HostNotificationSetting>> batch) {
        batchInsert(INSERT_VALUE_MAPPER, batch).execute();
    }

    public void updateMode(WebmasterUser user, WebmasterHostId hostId, NotificationType type, NotificationChannel channel, HostNotificationMode mode) {
        updateMode(user, Collections.singleton(hostId), type, channel, mode);
    }

    public void updateMode(WebmasterUser user, Collection<WebmasterHostId> hostIds, NotificationType type, NotificationChannel channel,
                           HostNotificationMode mode) {
        if (mode == HostNotificationMode.DEFAULT) {
            delete()
                    .where(F.USER_ID.eq(user.getUserId()))
                    .and(F.HOST_ID.in(hostIds))
                    .and(F.TYPE.eq(type))
                    .and(F.CHANNEL.eq(channel))
                    .execute();
        } else {
            batchInsert(hostIds.stream().map(s -> Pair.of(user.getUserId(), new HostNotificationSetting(s, type, channel, mode))).collect(Collectors.toList()));
        }
    }

    public List<HostNotificationSetting> getSettings(WebmasterUser user, List<WebmasterHostId> hostIds) {
        return select(MAPPER)
                .where(F.USER_ID.eq(user.getUserId()))
                .and(F.HOST_ID.in(hostIds))
                .queryForList(
                        FieldMapper.create(F.HOST_ID, HostNotificationSetting::getHostId),
                        FieldMapper.create(F.TYPE, HostNotificationSetting::getType),
                        FieldMapper.create(F.CHANNEL, HostNotificationSetting::getChannel)
                );
    }

    public void insert(WebmasterUser user, List<HostNotificationSetting> settings) {
        batchInsert(settings.stream().map(s -> Pair.of(user.getUserId(), s)).collect(Collectors.toList()));
    }

    public List<HostNotificationSetting> getSettings(WebmasterUser user) {
        return select(MAPPER)
                .where(F.USER_ID.eq(user.getUserId()))
                .queryForList();
    }


    public Map<Long, List<HostNotificationSetting>> listSettingsForUsers(Collection<Long> userIds) {
        final List<Pair<Long, HostNotificationSetting>> pairs = select(LIST_DATA_MAPPER)
                .where(F.USER_ID.in(userIds))
                .queryForList(
                        FieldMapper.create(F.USER_ID, Pair::getLeft),
                        FieldMapper.create(F.HOST_ID, p -> p.getRight().getHostId()),
                        FieldMapper.create(F.TYPE, p -> p.getRight().getType()),
                        FieldMapper.create(F.CHANNEL, p -> p.getRight().getChannel())
                );
        return pairs.stream().collect(W3Collectors.groupByPair());
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private static final DataMapper<HostNotificationSetting> MAPPER = DataMapper.create(
            F.HOST_ID, F.TYPE, F.CHANNEL, F.MODE,
            HostNotificationSetting::new
    );

    private static final DataMapper<Pair<Long, HostNotificationSetting>> LIST_DATA_MAPPER =
            DataMapper.create(F.USER_ID, MAPPER, Pair::of);

    private interface F {
        Field<Long> USER_ID = Fields.longField("user_id");
        Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        Field<NotificationType> TYPE = Fields.intEnumField("type", NotificationType.R);
        Field<NotificationChannel> CHANNEL = Fields.intEnumField("channel", NotificationChannel.R);
        Field<HostNotificationMode> MODE = Fields.intEnumField("mode", HostNotificationMode.R);
    }
}
