package ru.yandex.webmaster3.storage.user.dao;

import java.util.List;
import java.util.function.Consumer;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.sitestructure.SiteTreeNode;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;

/**
 * <p>Data access object for user nodes of site structure. Nodes are added and removed by user and are exported to
 * yt yandex map reduce server in order to collect indexing statistics for them.
 * <p>
 * Invariants in database:
 * <ul>
 * <li>user node name starts with slash
 * <li>user node node_id equals to hashCode of concatenation of strings &quot;user::&quot; and node name
 * </ul>
 */
@Repository
public class UserSiteTreeYDao extends AbstractYDao {
    public static final String TABLE_NAME = "user_site_tree";
    private static final long ROOT_NODE_ID = SiteTreeNode.createNodeId("/", false);

    public UserSiteTreeYDao() {
        super(PREFIX_USER, TABLE_NAME);
    }

    public SiteTreeNode addUserNode(final WebmasterHostId hostId, final String nodeName, final String originalName) {
        final long nodeId = SiteTreeNode.createNodeId(nodeName, true);
        upsert(
                F.HOST_ID.value(hostId),
                F.NODE_ID.value(nodeId),
                F.NAME.value(nodeName),
                F.ORIGINAL_NAME.value(originalName)
        ).execute();
        return new SiteTreeNode(nodeId, ROOT_NODE_ID, nodeName, originalName, null, null, null, true, null);
    }

    public void deleteUserNode(WebmasterHostId hostId, long nodeId) {
        delete()
                .where(F.HOST_ID.eq(hostId))
                .and(F.NODE_ID.eq(nodeId))
                .execute();
    }

    public List<SiteTreeNode> getUserTreeNodes(WebmasterHostId hostId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .queryForList(Pair.of(F.NODE_ID, SiteTreeNode::getNodeId));
    }

    public SiteTreeNode getUserTreeNode(WebmasterHostId hostId, long nodeId) {
        return select(MAPPER)
                .where(F.HOST_ID.eq(hostId))
                .and(F.NODE_ID.eq(nodeId))
                .queryOne();
    }

    public void forEach(Consumer<Pair<WebmasterHostId, String>> pairHostIdName) {
        streamReader(PAIR_MAPPER, pairHostIdName, true);
    }

    private static final DataMapper<Pair<WebmasterHostId, String>> PAIR_MAPPER = DataMapper.create(
            F.HOST_ID, F.NAME, Pair::of
    );

    private static final DataMapper<SiteTreeNode> MAPPER = DataMapper.create(
            F.NODE_ID, F.NAME, F.ORIGINAL_NAME, (nodeId, name, origName) -> {
                if (origName == null) {
                    origName = name;
                }
                return new SiteTreeNode(nodeId, ROOT_NODE_ID, name, origName, null, null, null, true, null);
            }
    );

    private static class F {
        static final Field<WebmasterHostId> HOST_ID = Fields.hostIdField("host_id");
        static final Field<Long> NODE_ID = Fields.longField("node_id");
        static final Field<String> NAME = Fields.stringField("name");
        static final Field<String> ORIGINAL_NAME = Fields.stringField("orig_name").makeOptional();
    }
}
