package ru.yandex.webmaster3.storage.user.dao;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;


@Slf4j
@Repository
public class UserUnreadMessagesCacheYDao extends AbstractYDao {

    private static final String TABLE_NAME = "user_unread_messages_cache";

    private static final Duration TTL = Duration.standardMinutes(10);

    public UserUnreadMessagesCacheYDao() {
        super(PREFIX_CACHE, TABLE_NAME);
    }

    public void setValue(long userId, long value)  {
        try {
            upsert(F.USER_ID.value(userId), F.VALUE.value(value), F.TTL_DATE.value(DateTime.now())).execute();
        } catch (WebmasterYdbException e) {
            log.warn("Error setting record to Ydb", e);
        }
    }

    public Long getValue(long userId) {
        try {
            DateTime ttlThreshold = DateTime.now().minus(TTL);

            return select(F.VALUE)
                    .where(F.USER_ID.eq(userId))
                    .and(F.TTL_DATE.gte(ttlThreshold))
                    .queryOne();
        } catch (WebmasterYdbException e) {
            log.warn("Error retrieving record from Ydb", e);
            return null;
        }
    }

    public void invalidate(long userId) {
        try {
            delete().where(F.USER_ID.eq(userId)).execute();
        } catch (WebmasterYdbException e) {
            log.warn("Error deleting record from Ydb", e);
        }
    }

    public void deleteForUser(long userId) {
        delete().where(F.USER_ID.eq(userId)).execute();
    }

    private interface F {
        Field<Long> USER_ID = Fields.longField("user_id");
        Field<Long> VALUE = Fields.longField("value");
        Field<DateTime> TTL_DATE = Fields.jodaDateTimeField("ttl_date");
    }
}
