package ru.yandex.webmaster3.storage.user.service;

import com.google.common.primitives.Longs;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableLong;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.util.trie.*;
import ru.yandex.webmaster3.storage.user.dao.InitializedUsersYDao;
import ru.yandex.webmaster3.storage.user.dao.VerifiedHostsYDao;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author leonidrom
 */
@Service
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class AllVerifiedHostUsersCacheService {
    private static final long CACHE_REFRESH_TIME_MS = TimeUnit.HOURS.toMillis(1);
    private static final UUIDDriver UUID_DRIVER = new UUIDDriver();
    private static final long TRIE_BATCH_SIZE = 200000;

    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final VerifiedHostsYDao verifiedHostsYDao;
    private final InitializedUsersYDao initializedUsersYDao;
    private volatile CompactTrieMap<Long, UUID> trie = null;

    public void init() {
        executor.scheduleAtFixedRate(
                this::resolveTrie,
                0,
                CACHE_REFRESH_TIME_MS,
                TimeUnit.MILLISECONDS
        );
    }

    public void destroy() {
        executor.shutdownNow();
    }

    @Nullable
    public UUID getUserUUIDNoBlock(long userId) {
        if (trie == null) {
            return null;
        }

        return trie.get(userId);
    }

    private void resolveTrie() {
        log.info("Trie is stale, update...");
        long updateStart = System.currentTimeMillis();
        // специально не освобождаем память, что бы после инициализации trie всегда был доступен
        trie = new CompactTrieMap<>(buildTrie(), USER_ID_CONVERTER);
        log.info("Trie loaded: {} bytes, load time {}",
                trie.getDataSizeBytes(),
                System.currentTimeMillis() - updateStart);
    }


    private CompactTrie<UUID> buildTrie() throws WebmasterException {
        MutableObject<CompactTrie<UUID>> compactTrie = new MutableObject<>();
        MutableObject<CompactTrieBuilder<UUID>> trie = new MutableObject<>(new CompactTrieBuilder<>());
        MutableLong users = new MutableLong();
        MutableLong curUserId = new MutableLong(0);
        MutableLong sumLen = new MutableLong();
        long start = System.currentTimeMillis();
        try {
            List<Long> batch = new ArrayList<>();
            verifiedHostsYDao.forEachUser(userId -> {
                if (userId == curUserId.longValue()) {
                    return;
                }

                curUserId.setValue(userId);
                batch.add(userId);
                if (batch.size() == TRIE_BATCH_SIZE / 10) {
                    sumLen.add(addToTrie(trie.getValue(), batch));
                    batch.clear();
                }

                if (users.incrementAndGet() % TRIE_BATCH_SIZE == 0) {
                    log.info("Loaded {} users", users.getValue());
                    long startMerge = System.currentTimeMillis();
                    CompactTrie<UUID> newTrie;
                    if (compactTrie.getValue() == null) {
                        newTrie = CompactTrie.fromNode(trie.getValue(), UUID_DRIVER);
                    } else {
                        newTrie = CompactTrie.fromNode(MergedNodeImpl.createMerged(compactTrie.getValue().getRootNode(), trie.getValue().getRoot(), (a, b) -> b), UUID_DRIVER);
                    }
                    trie.setValue(new CompactTrieBuilder<>());
                    compactTrie.setValue(null);
                    newTrie = newTrie.shrink();
                    compactTrie.setValue(newTrie);
                    log.info("Merged in {}ms", System.currentTimeMillis() - startMerge);
                }
            });

            sumLen.add(addToTrie(trie.getValue(), batch));
            if (compactTrie.getValue() == null) {
                compactTrie.setValue(CompactTrie.fromNode(trie.getValue(), UUID_DRIVER).shrink());
            } else {
                compactTrie.setValue(CompactTrie.fromNode(MergedNodeImpl.createMerged(compactTrie.getValue().getRootNode(), trie.getValue().getRoot(), (a, b) -> b), UUID_DRIVER).shrink());
            }
            long loaded = System.currentTimeMillis();
            log.info("Loaded in {} seconds, sum length {}", (loaded - start) / 1000, sumLen.getValue());
        } catch (Exception e) {
            log.error("Exception", e);
            throw new RuntimeException(e);
        }

        return compactTrie.getValue();
    }

    private long addToTrie(CompactTrieBuilder<UUID> trie, List<Long> userIds) {
        var userInfos = initializedUsersYDao.getUserInfos(userIds);
        long sum = 0;
        for (var userInfo : userInfos) {
            byte[] key = USER_ID_CONVERTER.toRawData(userInfo.userId());
            trie.insert(key, userInfo.userUuid());
            sum += key.length;
        }

        return sum;
    }

    private static final IDataConverter<Long> USER_ID_CONVERTER = new IDataConverter<>() {
        @Override
        public byte[] toRawData(Long userId) {
            return Longs.toByteArray(userId);
        }

        @Override
        public Long fromRawData(byte[] data, int len) {
            return Longs.fromByteArray(data);
        }
    };
}
