package ru.yandex.webmaster3.storage.user.service;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.user.ServiceAnnouncement;
import ru.yandex.webmaster3.storage.user.dao.ServiceAnnouncementYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

@Service("announcementService")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class AnnouncementService {
    private static final long CACHE_DURATION_MINUTES = 30;
    private static final long REFRESH_DURATION_MINUTES = 2;
    private static final Object KEY = new Object();

    private final AbtService abtService;
    private final ServiceAnnouncementYDao serviceAnnouncementYDao;

    private final ExecutorService executor = Executors.newSingleThreadExecutor();

    private LoadingCache<Object, Set<ServiceAnnouncement>> cache = CacheBuilder.newBuilder()
            .expireAfterAccess(CACHE_DURATION_MINUTES, TimeUnit.MINUTES)
            .refreshAfterWrite(REFRESH_DURATION_MINUTES, TimeUnit.MINUTES)
            .build(new CacheLoader<>() {
                @Override
                public Set<ServiceAnnouncement> load(@NotNull Object key) {
                    return new HashSet<>(serviceAnnouncementYDao.getServiceAnnouncements());
                }

                @Override
                public ListenableFuture<Set<ServiceAnnouncement>> reload(@NonNull Object key, Set<ServiceAnnouncement> oldValue) {
                    final var task = ListenableFutureTask.create(() -> {
                                try {
                                    return new HashSet<>(serviceAnnouncementYDao.getServiceAnnouncements());
                                } catch (WebmasterYdbException e) {
                                    return oldValue;
                                }
                            }
                    );
                    executor.submit(task);
                    return task;
                }
            });

    @NonNull
    public Set<ServiceAnnouncement> getServiceAnnouncementByRouteNameRegexp(String routeName, boolean onlyEnabled) {
        return cache.getUnchecked(KEY).stream()
                .filter(announcement -> !onlyEnabled || announcement.isEnabled())
                .filter(announcement -> routeName.matches(announcement.getRouteNameRegexp())).collect(Collectors.toSet());
    }

    @NonNull
    public Set<ServiceAnnouncement> getServiceAnnouncements(String routeName, boolean onlyEnabled, WebmasterHostId hostId, long userId) {
        return cache.getUnchecked(KEY).stream()
                .filter(announcement -> !onlyEnabled || announcement.isEnabled())
                .filter(announcement -> routeName.matches(announcement.getRouteNameRegexp()))
                .filter(announcement -> Strings.isNullOrEmpty(announcement.getExperiment()) || // либо нет экспериментов
                        (hostId != null && abtService.isInExperiment(hostId, announcement.getExperiment())) ||
                        (abtService.isInExperiment(userId, announcement.getExperiment()))
                )
                .map(announcement -> substituteParams(hostId, announcement, userId))
                .collect(Collectors.toSet());
    }

    private ServiceAnnouncement substituteParams(WebmasterHostId hostId, ServiceAnnouncement serviceAnnouncement, long userId) {
        if (hostId == null) {
            return serviceAnnouncement;
        }

        final Map<String, String> values = Map.of(
                "hostId", hostId.toString(),
                "host", IdUtils.hostIdToUrl(hostId),
                "userId", String.valueOf(userId)
        );
        StrSubstitutor sub = new StrSubstitutor(values);

        final Map<LanguageEnum, String> newTexts = serviceAnnouncement.getTexts().entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, x -> sub.replace(x.getValue())));

        return serviceAnnouncement.withTexts(newTexts);

    }


    public void setServiceAnnouncement(String routeName, Map<LanguageEnum, String> texts, boolean enabled, ServiceAnnouncement.Type type) {
        final ServiceAnnouncement announcement = new ServiceAnnouncement(routeName, texts, enabled, null, type);
        serviceAnnouncementYDao.insert(announcement);
    }
}
