package ru.yandex.webmaster3.storage.user.service;

import java.util.*;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.dao.UserMessageUpdatesYDao;
import ru.yandex.webmaster3.storage.user.dao.UserMessages3CHDao;
import ru.yandex.webmaster3.storage.user.dao.UserMessages4CHDao;
import ru.yandex.webmaster3.storage.user.dao.UserUnreadMessagesCacheYDao;
import ru.yandex.webmaster3.storage.user.message.MessageId;
import ru.yandex.webmaster3.storage.user.message.MessageUpdates;
import ru.yandex.webmaster3.storage.user.message.MessagesFilter;
import ru.yandex.webmaster3.storage.user.message.UserMessageInfo;

/**
 * Created by ifilippov5 on 13.04.17.
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class NewUserMessagesService implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(NewUserMessagesService.class);
    public static final int BATCH_SIZE = 1_000;

    private final UserUnreadMessagesCacheYDao userUnreadMessagesCacheYDao;
    private final UserMessageUpdatesYDao userMessageUpdatesYDao;
    private final UserMessages3CHDao mdbUserMessages3CHDao;
    private final UserMessages4CHDao mdbUserMessages4CHDao;
    private final InitializedUsersService initializedUsersService;
    private final AbtService abtService;

    private static final MessagesFilter EMPTY_MESSAGE_FILTER =
            new MessagesFilter(false, false, null);

    public UserMessageInfo getMessage(long userId, MessageId id) {
        UUID userUUID = null;
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            userUUID = initializedUsersService.getUUIDById(userId);
            if (userUUID == null) {
                log.error("No UUID for user {}", userId);
                return null;
            }
        }

        UserMessageInfo result = null;
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            try {
                result = mdbUserMessages4CHDao.getMessage(userUUID, id);
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }
        } else {
            result = mdbUserMessages3CHDao.getMessage(userId, id);
        }

        if (result != null) {
            MessageUpdates updates = userMessageUpdatesYDao.getUpdatesForMessage(userId, id);
            if (updates != null) {
                if (updates.isDeleted()) {
                    return null;
                }
                if (updates.isRead() != null) {
                    result = UserMessageInfo.createWithRead(result, updates.isRead());
                }
            }
        }

        return result;
    }

    public Pair<MessageId, MessageId> getPreviousAndNextMessageId(long userId, UserMessageInfo messageInfo,
                                                                  MessagesFilter filter) {
        MessageId prevMessageId;
        MessageId nextMessageId;
        List<MessageUpdates> updates = userMessageUpdatesYDao.getUpdatesForUser(userId);
        Set<MessageId> deletedIds = updates.stream()
                .filter(MessageUpdates::isDeleted)
                .map(MessageUpdates::getMessageId)
                .collect(Collectors.toSet());

        prevMessageId = mdbUserMessages3CHDao.getNeighbourIdByFilter(messageInfo, deletedIds, filter, false);
        nextMessageId = mdbUserMessages3CHDao.getNeighbourIdByFilter(messageInfo, deletedIds, filter, true);
        return Pair.of(prevMessageId, nextMessageId);
    }

    public Pair<List<UserMessageInfo>, Long> listAndCountMessages(long userId, MessagesFilter filter,
                                                                  PageUtils.Pager pager) {
        UUID userUUID = null;
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            userUUID = initializedUsersService.getUUIDById(userId);
            if (userUUID == null) {
                log.error("No UUID for user {}", userId);
                return Pair.of(Collections.emptyList(), 0L);
            }
        }

        List<MessageUpdates> updates = userMessageUpdatesYDao.getUpdatesForUser(userId);
        Set<MessageId> deletedIds =
                updates.stream().filter(MessageUpdates::isDeleted).map(MessageUpdates::getMessageId).collect(Collectors.toSet());
        Map<MessageId, Boolean> readStates = updates.stream()
                .filter(mu -> mu.isRead() != null)
                .collect(Collectors.toMap(MessageUpdates::getMessageId, MessageUpdates::isRead));

        List<UserMessageInfo> messageInfos = Collections.emptyList();
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            try {
                messageInfos = mdbUserMessages4CHDao.listMessages(userUUID, deletedIds, filter, pager);
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }
        } else {
            messageInfos = mdbUserMessages3CHDao.listMessages(userId, deletedIds, filter, pager);
        }

        messageInfos = messageInfos.stream()
                .map(message -> UserMessageInfo.createWithRead(message, readStates.getOrDefault(message.getId(),
                        message.isRead())))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        long count = 0;
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            try {
                count = mdbUserMessages4CHDao.count(userUUID, deletedIds, filter);
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }
        } else {
            count = mdbUserMessages3CHDao.count(userId, deletedIds, filter);
        }

        return Pair.of(messageInfos, count);
    }

    public void deleteAllMessages(long userId) {
        long count = count(userId, EMPTY_MESSAGE_FILTER);

        if (count >= 30) {
            long version = System.currentTimeMillis();
            mdbUserMessages3CHDao.deleteAllMessagesForUser(userId, version);
            try {
                var userUUID = initializedUsersService.getUUIDById(userId);
                if (userUUID != null) {
                    mdbUserMessages4CHDao.deleteAllMessagesForUser(userUUID, version);
                } else {
                    log.error("No UUID for user {}", userId);
                }
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }

            userUnreadMessagesCacheYDao.invalidate(userId);
        } else {
            List<MessageId> keys = listAndCountMessages(userId, EMPTY_MESSAGE_FILTER, null).getLeft().stream()
                    .map(UserMessageInfo::getId)
                    .collect(Collectors.toList());
            deleteMessages(userId, keys);
        }
    }

    public void markReadAll(long userId, boolean read) {
        long count = count(userId, EMPTY_MESSAGE_FILTER);
        if (count >= 30) {
            long version = System.currentTimeMillis();
            mdbUserMessages3CHDao.markReadAllMessagesForUser(userId, read, version);
            try {
                var userUUID = initializedUsersService.getUUIDById(userId);
                if (userUUID != null) {
                    mdbUserMessages4CHDao.markReadAllMessagesForUser(userUUID, read, version);
                } else {
                    log.error("No UUID for user {}", userId);
                }
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }

            userMessageUpdatesYDao.cleanReadUpdatesForUser(userId);
            userUnreadMessagesCacheYDao.invalidate(userId);
        } else {
            List<MessageId> keys = listAndCountMessages(userId, EMPTY_MESSAGE_FILTER, null).getLeft().stream()
                    .map(UserMessageInfo::getId)
                    .collect(Collectors.toList());
            markRead(userId, keys, read);
        }
    }

    public long unreadCount(long userId, boolean invalidateCache) {
        Long value = null;
        if (!invalidateCache) {
            value = userUnreadMessagesCacheYDao.getValue(userId);
        }
        MessagesFilter filter = new MessagesFilter(false, true, null);
        if (value == null) {
            value = count(userId, filter);
            userUnreadMessagesCacheYDao.setValue(userId, value);
        }
        return value;
    }

    public long count(long userId, MessagesFilter filter) {
        List<MessageUpdates> updates = userMessageUpdatesYDao.getUpdatesForUser(userId);
        Set<MessageId> excludeIds = updates.stream()
                .filter(update -> update.isDeleted() || !filter.isOnlyUnread() || Boolean.TRUE.equals(update.isRead()))
                .map(MessageUpdates::getMessageId).collect(Collectors.toSet());

        long count = 0;
        if (abtService.isInExperiment(userId, "NEW_USER_MESSAGES_READ")) {
            var userUUID = initializedUsersService.getUUIDById(userId);
            if (userUUID != null) {
                try {
                    count = mdbUserMessages4CHDao.count(userUUID, excludeIds, filter);
                } catch (Exception e) {
                    log.error("NEW_USER_MESSAGES error", e);
                }
            } else {
                log.error("No UUID for user {}", userId);
            }
        }  else {
            count = mdbUserMessages3CHDao.count(userId, excludeIds, filter);
        }

        return count;
    }

    public void deleteMessages(long userId, List<MessageId> messageIds) {
        if (messageIds.isEmpty()) {
            return;
        }

        var userUUID = initializedUsersService.getUUIDById(userId);
        List<UserMessageInfo> messages = messageIds.stream()
                .map(id -> UserMessageInfo.createDeleted(id, userId, userUUID))
                .collect(Collectors.toList());

        long version = System.currentTimeMillis();
        mdbUserMessages3CHDao.updateMessages(messages, version);
        if (userUUID != null) {
            try {
                mdbUserMessages4CHDao.updateMessages(messages, version);
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }
        } else {
            log.error("No UUID for user {}", userId);
        }

        List<MessageId> batch = new ArrayList<>();
        messageIds.forEach(messageId -> {
            batch.add(messageId);
            if (batch.size() >= BATCH_SIZE) {
                userMessageUpdatesYDao.batchMarkDeleted(userId, batch);
                batch.clear();
            }
        });
        userMessageUpdatesYDao.batchMarkDeleted(userId, batch);

        userUnreadMessagesCacheYDao.invalidate(userId);
    }

    public void markRead(long userId, List<MessageId> messageIds, boolean read) {
        if (messageIds.isEmpty()) {
            return;
        }

        var uuid = initializedUsersService.getUUIDById(userId);
        List<UserMessageInfo> messages = messageIds.stream()
                .map(id -> getMessage(userId, id))
                .filter(Objects::nonNull)
                .filter(message -> message.isRead() != read)
                .map(message -> UserMessageInfo.createWithRead(message, read))
                .map(messageInfo -> messageInfo.withUUID(uuid))
                .collect(Collectors.toList());

        long version = System.currentTimeMillis();

        mdbUserMessages3CHDao.updateMessages(messages, version);
        if (uuid != null) {
            try {
                mdbUserMessages4CHDao.updateMessages(messages, version);
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }
        } else {
            log.error("No UUID for user {}", userId);
        }

        List<MessageId> batch = new ArrayList<>();
        messages.forEach(
                messageInfo -> {
                    var id = messageInfo.getId();
                    batch.add(id);
                    if (batch.size() >= BATCH_SIZE) {
                        userMessageUpdatesYDao.batchMarkRead(userId, batch, read);
                        batch.clear();
                    }
                }
        );
        userMessageUpdatesYDao.batchMarkRead(userId, batch, read);

        userUnreadMessagesCacheYDao.invalidate(userId);
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        long userId = user.getUserId();
        userUnreadMessagesCacheYDao.deleteForUser(userId);
        userMessageUpdatesYDao.deleteForUser(userId);
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                userUnreadMessagesCacheYDao.getTablePath(),
                userMessageUpdatesYDao.getTablePath()
        );
    }
}
