package ru.yandex.webmaster3.storage.user.service;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.user.ServiceSurvey;
import ru.yandex.webmaster3.storage.user.dao.ServiceSurveyYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

@Service("surveyService")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SurveyService {
    private static final long CACHE_DURATION_MINUTES = 30;
    private static final long REFRESH_DURATION_MINUTES = 2;
    private static final Object KEY = new Object();

    private final AbtService abtService;
    private final ServiceSurveyYDao serviceSurveyYDao;

    private final ExecutorService executor = Executors.newSingleThreadExecutor();

    private final LoadingCache<Object, Set<ServiceSurvey>> cache = CacheBuilder.newBuilder()
            .expireAfterAccess(CACHE_DURATION_MINUTES, TimeUnit.MINUTES)
            .refreshAfterWrite(REFRESH_DURATION_MINUTES, TimeUnit.MINUTES)
            .build(new CacheLoader<>() {
                @Override
                public Set<ServiceSurvey> load(@NotNull Object key) {
                    return new HashSet<>(serviceSurveyYDao.getServiceSurveys());
                }

                @Override
                public ListenableFuture<Set<ServiceSurvey>> reload(@NonNull Object key, Set<ServiceSurvey> oldValue) {
                    final var task = ListenableFutureTask.create(() -> {
                                try {
                                    return new HashSet<>(serviceSurveyYDao.getServiceSurveys());
                                } catch (WebmasterYdbException e) {
                                    return oldValue;
                                }
                            }
                    );
                    executor.submit(task);
                    return task;
                }
            });

    @NonNull
    public Set<ServiceSurvey> getServiceSurveys(String routeName,
                                                boolean onlyEnabled,
                                                WebmasterHostId hostId,
                                                long userId) {
        return cache.getUnchecked(KEY).stream()
                .filter(survey -> !onlyEnabled || survey.isEnabled())
                .filter(survey -> routeName.matches(survey.getRouteNameRegexp()))
                .filter(survey -> Strings.isNullOrEmpty(survey.getExperiment()) || // либо нет экспериментов
                        (hostId != null && abtService.isInExperiment(hostId, survey.getExperiment())) ||
                        (abtService.isInExperiment(userId, survey.getExperiment()))
                )
                .map(survey -> substituteParams(hostId, survey, userId))
                .collect(Collectors.toSet());
    }

    private ServiceSurvey substituteParams(WebmasterHostId hostId, ServiceSurvey serviceSurvey, long userId) {
        if (hostId == null) {
            return serviceSurvey;
        }

        final Map<String, String> values = Map.of(
                "hostId", hostId.toString(),
                "host", IdUtils.hostIdToUrl(hostId),
                "userId", String.valueOf(userId)
        );
        StrSubstitutor sub = new StrSubstitutor(values);

        final Map<String, Map<LanguageEnum, String>> newTexts = serviceSurvey
                .getTexts()
                .entrySet()
                .stream()
                .collect(Collectors.toMap(
                        Map.Entry::getKey,
                        x -> x.getValue()
                                .entrySet()
                                .stream()
                                .collect(Collectors.toMap(
                                        Map.Entry::getKey,
                                        y -> sub.replace(y.getValue())
                                ))
                ));

        return serviceSurvey.withTexts(newTexts);

    }
}
