package ru.yandex.webmaster3.storage.user.service;

import java.util.List;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.user.SurveyState;
import ru.yandex.webmaster3.storage.user.SurveyStateEnum;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.dao.SurveyStateYDao;


@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Slf4j
public class SurveyStateService implements UserTakeoutDataProvider {

    private final SurveyStateYDao surveyStateForUserYDao;

    public boolean isClosed(String type, long userId, Integer daysBeforeReopen, WebmasterHostId hostId,
                            boolean byHost) {
        SurveyState surveyState = surveyStateForUserYDao.getState(type, userId, hostId, byHost);
        // первый заход, покажем опрос
        if (surveyState == null) {
            return false;
        }
        // если уже был ответ, то точно нельзя показывать
        if (surveyState.getState() == SurveyStateEnum.SUBMIT) {
            return true;
        }
        // если таймер не поставлен, не покажем опрос
        if (daysBeforeReopen == null) {
            return true;
        }
        // surveyState.lastPing + daysBeforeReopen > now
        return surveyState.getDate().isAfter(DateTime.now().minusDays(daysBeforeReopen));
    }

    public SurveyState setStatus(String type, long userId, SurveyStateEnum state, WebmasterHostId hostId) {
        SurveyState surveyStateForUser = new SurveyState(
                type,
                userId,
                hostId,
                state,
                DateTime.now()
        );
        surveyStateForUserYDao.insert(surveyStateForUser);
        return surveyStateForUser;
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        surveyStateForUserYDao.deleteForUser(user.getUserId());
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(surveyStateForUserYDao.getTablePath());
    }
}
