package ru.yandex.webmaster3.storage.user.service;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.blackbox.UserWithLogin;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.delegation.*;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.UserTakeoutTableData;

import java.util.ArrayList;
import java.util.List;

/**
 * @author leonidrom
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class UserDelegationsService implements UserTakeoutDataProvider {
    private static final String USER_TAKEOUT_DELEGATED_HOSTS_LABEL = "delegatedHosts";

    private final UserHostDelegationsYDao userHostDelegationsYDao;
    private final UserDailyDelegationsYDao userDailyDelegationsYDao;
    private final UserDelegationsForSendYDao userDelegationsForSendYDao;
    private final BlackboxUsersService blackboxExternalYandexUsersService;

    @Override
    public @NotNull List<UserTakeoutTableData> getUserTakeoutData(WebmasterUser user) {
        List<UserTakeoutTableData> takeoutData = new ArrayList<>();
        List<HostDelegationTakeoutRecord> delegationRecords = new ArrayList<>();
        userHostDelegationsYDao.forEach(r -> {
            if (r.getFromUser() == user.getUserId() || (r.getCancelledByUser() != null && r.getCancelledByUser() == user.getUserId())) {
                delegationRecords.add(toTakeoutRecord(r));
            }
        });
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_DELEGATED_HOSTS_LABEL, delegationRecords));

        return takeoutData;
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        userHostDelegationsYDao.deleteForFromUser(user);
        userHostDelegationsYDao.deleteForToUser(user);
        userDailyDelegationsYDao.deleteForUser(user);
        userDelegationsForSendYDao.delete(List.of(user.getUserId()));
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                userHostDelegationsYDao.getTablePath(),
                userDailyDelegationsYDao.getTablePath(),
                userDelegationsForSendYDao.getTablePath()
        );
    }

    private HostDelegationTakeoutRecord toTakeoutRecord(HostDelegationRecord record) {
        UserWithLogin toUser = blackboxExternalYandexUsersService.getUserById(record.getToUser());
        String toUserLogin = toUser == null ? null : toUser.getLogin();

        String canceledByUserLogin = null;
        if (record.getCancelledByUser() != null) {
            UserWithLogin canceledByUser = blackboxExternalYandexUsersService.getUserById(record.getCancelledByUser());
            if (canceledByUser != null) {
                canceledByUserLogin = canceledByUser.getLogin();
            }
        }

        return record.toTakeoutRecord(toUserLogin, canceledByUserLogin);
    }
}
