package ru.yandex.webmaster3.storage.user.service;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.*;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.UserTakeoutDataProvider;
import ru.yandex.webmaster3.storage.user.UserTakeoutTableData;
import ru.yandex.webmaster3.storage.user.dao.LoginsToUidsYDao;
import ru.yandex.webmaster3.storage.user.dao.PersonalInfoCacheYDao;
import ru.yandex.webmaster3.storage.user.dao.UserLastVisitYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
@Component
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class UserPersonalInfoService implements UserTakeoutDataProvider {
    private static final Logger log = LoggerFactory.getLogger(UserPersonalInfoService.class);

    private static final Duration PERSONAL_INFO_TTL = Duration.standardDays(2);
    private static final String USER_TAKEOUT_LAST_VISIT = "lastVisit";
    private static final String USER_TAKEOUT_USER_INFO = "userInfo";

    private static final Set<BlackboxAttributeType> ATTRIBUTES = EnumSet.of(
            BlackboxAttributeType.LOGIN,
//            BlackboxAttributeType.FIO,
            BlackboxAttributeType.LANG
    );

    private final BlackboxUsersService blackboxExternalYandexUsersService;
    private final LoginsToUidsYDao loginsToUidsYDao;
    private final PersonalInfoCacheYDao personalInfoCacheYDao;
    private final UserLastVisitYDao userLastVisitYDao;

    public Map<Long, UserPersonalInfo> resolveCacheMisses(Collection<Long> unknownUids, Collection<UserWithLogin> unknownPersonals) {
        try {
            Set<Long> bbReq = new HashSet<>(unknownUids);
            bbReq.addAll(unknownPersonals.stream().map(UserWithLogin::getUserId).collect(Collectors.toList()));
            if (bbReq.isEmpty()) {
                return Collections.emptyMap();
            }
            Map<Long, UserWithFields> users =
                    blackboxExternalYandexUsersService.findUsers(BlackboxQuery.byUids(bbReq), ATTRIBUTES, GetEmailsType.NONE)
                            .stream()
                            .collect(Collectors.toMap(UserWithFields::getUid, Function.identity()));
            if (users.size() < bbReq.size()) {
                for (long uid : bbReq) {
                    if (!users.containsKey(uid)) {
                        log.warn("User " + uid + " not found in blackbox");
                    }
                }
            }

            Map<Long, UserPersonalInfo> result = new HashMap<>();

            List<UserPersonalInfo> toFillUsersWithLogins = new ArrayList<>();

            for (long uid : unknownUids) {
                UserWithFields userWithFields = users.get(uid);
                if (userWithFields != null) {
                    toFillUsersWithLogins.add(new UserPersonalInfo(uid, userWithFields.getFields().get(BlackboxAttributeType.LOGIN)));
                    result.put(userWithFields.getUid(), UserPersonalInfo.fromUserWithFields(userWithFields));
                }
            }
            for (UserWithLogin user : unknownPersonals) {
                UserWithFields userWithFields = users.get(user.getUserId());
                if (userWithFields != null) {
                    result.put(user.getUserId(), UserPersonalInfo.fromUserWithFields(userWithFields));
                }
            }

            loginsToUidsYDao.addUsers(toFillUsersWithLogins.stream().map(u -> (UserWithLogin)u).toList());
            personalInfoCacheYDao.addUsers(toFillUsersWithLogins);
            personalInfoCacheYDao.updatePersonalInfos(result.values());

            return result;
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get users personal infos",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    private Map<String, UserPersonalInfo> getFromBlackbox(Collection<String> logins)  {
        Map<String, UserPersonalInfo> result = new HashMap<>();
        Map<String, Long> toFillLogins = new HashMap<>();
        for (String login : logins) {
            List<UserWithFields> users = blackboxExternalYandexUsersService.findUsers(BlackboxQuery.byLogin(login), ATTRIBUTES, GetEmailsType.NONE);
            if (!users.isEmpty()) {
                UserPersonalInfo user = UserPersonalInfo.fromUserWithFields(users.get(0));
                result.put(login, user);
                toFillLogins.put(login, user.getUserId());
                // В общем случае у пользователя могут быть разные логины - так что давайте возьмем "каноничный"
                toFillLogins.put(user.getLogin(), user.getUserId());
            }
        }
        {
            List<UserWithLogin> forLogin2UidMapping = new ArrayList<>();
            for (Map.Entry<String, Long> entry : toFillLogins.entrySet()) {
                forLogin2UidMapping.add(new UserWithLogin(entry.getValue(), entry.getKey()));
            }

            loginsToUidsYDao.addUsers(forLogin2UidMapping);
        }

        personalInfoCacheYDao.addUsers(result.values());
        personalInfoCacheYDao.updatePersonalInfos(result.values());

        return result;
    }

    @Nullable
    public UserPersonalInfo getUserPersonalInfo(long userId) {
        Map<Long, UserPersonalInfo> map = getUsersPersonalInfos(Collections.singleton(userId));
        return map.get(userId);
    }

    public Map<Long, UserPersonalInfo> getUsersPersonalInfos(Collection<Long> userIds) {
        CachedResponse response = getCachedUsersPersonalInfos(userIds);
        Map<Long, UserPersonalInfo> result = response.result;
        result.putAll(resolveCacheMisses(response.getUnknownUsers(), response.unknownPersonals));
        return result;
    }

    public CachedResponse getCachedUsersPersonalInfos(Collection<Long> userIds) {
        try {
            List<UserPersonalInfo> cacheInfos = personalInfoCacheYDao.getPersonalInfos(userIds);
            Map<Long, UserPersonalInfo> result = new HashMap<>();
            Set<Long> absentUsers = new HashSet<>();
            Map<Long, UserWithLogin> missingPersonalInfos = new HashMap<>();
            for (UserPersonalInfo personalInfo : cacheInfos) {
                if (personalInfo.isComplete()) {
                    result.put(personalInfo.getUserId(), personalInfo);
                } else {
                    missingPersonalInfos.put(
                            personalInfo.getUserId(),
                            new UserWithLogin(personalInfo.getUserId(), personalInfo.getLogin())
                    );
                }
            }
            for (long uid : userIds) {
                if (!missingPersonalInfos.containsKey(uid) && !result.containsKey(uid)) {
                    absentUsers.add(uid);
                }
            }
            log.info("{} users found by uids, {} users missing, {} users missing personal infos", result.size(), absentUsers.size(), missingPersonalInfos.size());

            return new CachedResponse(result, absentUsers, missingPersonalInfos.values());
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get users personal infos",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    public Map<String, UserPersonalInfo> getUserPersonalInfosByLogins(Collection<String> logins) {
        try {
            Map<String, Long> uids = loginsToUidsYDao.getUids(logins);
            Map<String, UserPersonalInfo> result = new HashMap<>();
            List<String> unknownLogins = new ArrayList<>();
            for (String login : logins) {
                if (!uids.containsKey(login)) {
                    unknownLogins.add(login);
                }
            }
            log.info("{} users found by logins, {} users missing", uids.size(), unknownLogins.size());
            if (!unknownLogins.isEmpty()) {
                result.putAll(getFromBlackbox(unknownLogins));
            }
            if (!uids.isEmpty()) {
                Map<Long, UserPersonalInfo> uid2info = getUsersPersonalInfos(uids.values());
                for (Map.Entry<String, Long> entry : uids.entrySet()) {
                    UserPersonalInfo personalInfo = uid2info.get(entry.getValue());
                    if (personalInfo != null) {
                        result.put(entry.getKey(), personalInfo);
                    }
                }
            }
            return result;
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get user personal infos by logins",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    @Override
    @NotNull
    public List<UserTakeoutTableData> getUserTakeoutData(WebmasterUser user) {
        List<UserTakeoutTableData> takeoutData = new ArrayList<>();

        UserPersonalInfo personalInfo = getUserPersonalInfo(user.getUserId());
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_USER_INFO, personalInfo));

        Instant lastVisit = userLastVisitYDao.getVisit(user.getUserId());
        takeoutData.add(new UserTakeoutTableData(USER_TAKEOUT_LAST_VISIT, lastVisit));

        return takeoutData;
    }

    @Override
    public void deleteUserData(WebmasterUser user) {
        long userId = user.getUserId();
        loginsToUidsYDao.deleteForUser(userId);
        userLastVisitYDao.deleteForUser(userId);
        personalInfoCacheYDao.deleteForUser(userId);
    }

    @Override
    public @NotNull List<String> getTakeoutTables() {
        return List.of(
                loginsToUidsYDao.getTablePath(),
                userLastVisitYDao.getTablePath(),
                personalInfoCacheYDao.getTablePath()
        );
    }

    public static class CachedResponse {
        private final Map<Long, UserPersonalInfo> result;
        private final Set<Long> unknownUsers;
        private final Collection<UserWithLogin> unknownPersonals;

        public CachedResponse(Map<Long, UserPersonalInfo> result, Set<Long> unknownUsers, Collection<UserWithLogin> unknownPersonals) {
            this.result = result;
            this.unknownUsers = unknownUsers;
            this.unknownPersonals = unknownPersonals;
        }

        public Map<Long, UserPersonalInfo> getResult() {
            return result;
        }

        public Set<Long> getUnknownUsers() {
            return unknownUsers;
        }

        public Collection<UserWithLogin> getUnknownPersonals() {
            return unknownPersonals;
        }
    }
}
