package ru.yandex.webmaster3.storage.user.service;

import java.util.*;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.core.JsonProcessingException;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.user.takeout.UserTakeoutRequestType;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.UserTakeoutTaskData;
import ru.yandex.webmaster3.storage.takeout.YtUserDataDeleteQueueYDao;
import ru.yandex.webmaster3.storage.takeout.YtUserDataDeleteService;
import ru.yandex.webmaster3.storage.user.*;
import ru.yandex.webmaster3.storage.user.dao.InitializedUsersYDao;
import ru.yandex.webmaster3.storage.takeout.UserTakeoutRequestsYDao;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

import javax.annotation.PostConstruct;

/**
 * @author leonidrom
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Service
public class UserTakeoutService {
    public static final Duration GET_RETRY_PERIOD = Duration.standardMinutes(30);
    public static final Duration DELETE_RETRY_PERIOD = Duration.standardHours(4);

    private static final Logger log = LoggerFactory.getLogger(UserTakeoutService.class);
    private static RetryUtils.RetryPolicy YDB_RETRY_POLICY = RetryUtils.linearBackoff(5, Duration.standardSeconds(30));

    private final UserTakeoutRequestsYDao userTakeoutRequestsYDao;
    private final YtUserDataDeleteService ytUserDataDeleteService;
    private final InitializedUsersYDao initializedUsersYDao;
    private final WorkerClient workerClient;
    private final ApplicationContext applicationContext;

    private List<UserTakeoutDataProvider> takeoutDataProviders;

    @PostConstruct
    public void init() {
        Map<String, UserTakeoutDataProvider> providersMap = applicationContext.getBeansOfType(UserTakeoutDataProvider.class);
        takeoutDataProviders = new ArrayList<>(providersMap.values());
    }

    /**
     * Создает запрос на выгрузку данных пользователя по GDPR.
     *
     * @param userId пользователь, для которого надо сделать выгрузку/удалить данные. Может не быть пользователем Вебмастера.
     * @return UUID созданого запроса на выгрузку
     */
    public UserTakeoutRequest createRequest(long userId, UserTakeoutRequestType type) {
        UserTakeoutRequest request;

        if (!isWebmasterUser(userId)) {
            // если это не пользователь Вебмастера, то отдавать/удалять нечего
            request = createAndSaveRequest(userId, UserTakeoutRequestStatus.DONE, type);
        } else {
            request = createAndSaveRequest(userId, UserTakeoutRequestStatus.IN_PROGRESS, type);
            workerClient.enqueueTask(new UserTakeoutTaskData(userId, request.getRequestId(), type));
        }

        return request;
    }

    private UserTakeoutRequest createAndSaveRequest(long userId, UserTakeoutRequestStatus status, UserTakeoutRequestType type) {
        UUID requestId = UUIDs.timeBased();
        UserTakeoutRequest request = new UserTakeoutRequest(requestId, userId, type, status);
        userTakeoutRequestsYDao.saveRequest(request);

        return request;
    }

    private void markRequestAsDone(UUID requestId, String data) {
        userTakeoutRequestsYDao.markAsDone(requestId, data);
    }

    public void markRequestAsDone(UUID requestId) {
        userTakeoutRequestsYDao.markAsDone(requestId);
    }

    public void markRequestAsFailed(UUID requestId) {
        userTakeoutRequestsYDao.markAsFailed(requestId);
    }

    @NotNull
    public UserTakeoutRequest getRequest(UUID requestId) {
        UserTakeoutRequest request = userTakeoutRequestsYDao.getRequest(requestId);
        if (request == null) {
            final String msg = "Request " + requestId + " is not found";
            log.error(msg);
            throw new WebmasterException(msg, new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(),
                    msg), null);
        }

        return request;
    }

    @Nullable
    public UserTakeoutRequest getRequest(long userId) {
        return userTakeoutRequestsYDao.getRequest(userId);
    }

    public void processGetRequest(UUID requestId, long userId) {
        UserTakeoutRequest request = getRequest(requestId);
        WebmasterUser user = new WebmasterUser(userId);
        if (request.getStatus().isTerminal()) {
            // уже обработали
            log.info("Skipping request {}, already processed", requestId);
            return;
        }

        // соберем все данные для выгрузки
        log.info("Collecting takeout data for {}", requestId);
        List<UserTakeoutTableData> takeoutData = collectTablesTakeoutData(user);
        log.info("Collected {} takeout entries for {}", takeoutData.size(), requestId);

        // сериализуем их в JSON строку
        String takeoutDataAsJSON;
        try {
            Map<String, Object> takeoutDataAsMap = new TreeMap<>();
            takeoutData.stream().map(UserTakeoutTableData::getAsMap).forEach(takeoutDataAsMap::putAll);
            takeoutDataAsJSON = JsonDBMapping.OM.writeValueAsString(takeoutDataAsMap);
        } catch (JsonProcessingException e) {
            String msg = "Failed to convert takeout data to JSON";
            log.error(msg, e);
            throw new WebmasterException(msg, new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(),
                    msg), e);
        }

        // и сохраним в базе
        markRequestAsDone(requestId, takeoutDataAsJSON);
    }

    public void processDeleteRequest(UUID requestId, long userId) {
        UserTakeoutRequest request = getRequest(requestId);
        if (request.getStatus().isTerminal()) {
            // уже обработали
            log.info("Skipping request {}, already processed", requestId);
            return;
        }

        deleteUserData(requestId, userId);
    }

    public List<String> getTakeoutTables() {
        List<String> takeoutTables = new ArrayList<>();
        takeoutDataProviders.forEach(p -> takeoutTables.addAll(p.getTakeoutTables()));
        return takeoutTables;
    }

    private List<UserTakeoutTableData> collectTablesTakeoutData(WebmasterUser user) {
        List<UserTakeoutTableData> takeoutData = new ArrayList<>();
        for (var p : takeoutDataProviders) {
            try {
                takeoutData.addAll(RetryUtils.query(YDB_RETRY_POLICY, () -> p.getUserTakeoutData(user)));
            } catch (InterruptedException e) {
                throw new WebmasterYdbException(e);
            }
        }

        return takeoutData;
    }

    private void deleteUserData(UUID requestId, long userId) {
        initializedUsersYDao.deleteForUser(userId);

        WebmasterUser user = new WebmasterUser(userId);
        for (var p : takeoutDataProviders) {
            try {
                log.info("Deleting from: {}", Arrays.toString(p.getTakeoutTables().toArray()));
                RetryUtils.execute(YDB_RETRY_POLICY, () -> {
                    p.deleteUserData(user);
                });
            } catch (InterruptedException e) {
                throw new WebmasterYdbException(e);
            }
        }

        // из YT специально удаляем в последнюю очередь
        ytUserDataDeleteService.deleteUserData(requestId, userId);
    }

    private boolean isWebmasterUser(long userId) {
        return initializedUsersYDao.getUserInfo(userId) != null;
    }
}
