package ru.yandex.webmaster3.storage.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.comparators.ComparatorChain;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.HostDomainInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.domain.HostDomainInfoService;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.user.UserUnverifiedHost;

/**
 * @author aherman
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostListTreeBuilder {
    private final MirrorService2 mirrorService2;
    private final HostDomainInfoService hostDomainInfoService;

    public Pair<List<HostTreeNode>, HostCounters> createHostTree(List<UserVerifiedHost> verifiedHosts,
                                                                 List<UserUnverifiedHost> unverifiedHosts,
                                                                 List<WebmasterHostId> pinnedHosts) {
        Map<WebmasterHostId, HostTreeNode> hostNodes = new HashMap<>();
        List<HostTreeNode> topLevelNodes = new ArrayList<>();
        final Set<WebmasterHostId> verifiedAndUnverifiedHosts = Stream.concat(
                verifiedHosts.stream().map(UserVerifiedHost::getWebmasterHostId),
                unverifiedHosts.stream().map(UserUnverifiedHost::getWebmasterHostId)).collect(Collectors.toSet());
        final Map<WebmasterHostId, WebmasterHostId> mainMirrors = mirrorService2.getMainMirrors(verifiedAndUnverifiedHosts);
        for (UserVerifiedHost verifiedHost : verifiedHosts) {
            WebmasterHostId webmasterHostId = verifiedHost.getWebmasterHostId();
            HostTreeNode node = hostNodes.get(webmasterHostId);
            boolean newlyCreated = false;
            if (node == null) {
                node = new HostTreeNode(webmasterHostId, true);
                hostNodes.put(webmasterHostId, node);
                newlyCreated = true;
            } else {
                node.setVerified(true);
                node.setAdded(true);
            }

            WebmasterHostId mainMirrorHostId = mainMirrors.get(webmasterHostId);
            if (mainMirrorHostId == null) {
                if (newlyCreated) {
                    topLevelNodes.add(node);
                }
            } else {
                HostTreeNode mainMirrorNode = hostNodes.get(mainMirrorHostId);
                if (mainMirrorNode == null) {
                    mainMirrorNode = new HostTreeNode(mainMirrorHostId);
                    hostNodes.put(mainMirrorHostId, mainMirrorNode);
                    topLevelNodes.add(mainMirrorNode);
                }
                mainMirrorNode.addChildren(node);
            }
        }

        int unverifiedHostCount = 0;

        for (UserUnverifiedHost unverifiedHost : unverifiedHosts) {
            WebmasterHostId webmasterHostId = unverifiedHost.getWebmasterHostId();
            HostTreeNode node = hostNodes.get(webmasterHostId);
            if (node != null) {
                node.setAdded(true);
                continue;
            }
            unverifiedHostCount++;
            node = new HostTreeNode(webmasterHostId, false);
            hostNodes.put(webmasterHostId, node);
            WebmasterHostId mainMirrorHostId = mainMirrors.get(webmasterHostId);
            if (mainMirrorHostId == null) {
                topLevelNodes.add(node);
            } else {
                HostTreeNode mainMirrorNode = hostNodes.get(mainMirrorHostId);
                if (mainMirrorNode == null) {
                    mainMirrorNode = new HostTreeNode(mainMirrorHostId);
                    hostNodes.put(mainMirrorHostId, mainMirrorNode);
                    topLevelNodes.add(mainMirrorNode);
                }
                mainMirrorNode.addChildren(node);
            }
        }

        for (WebmasterHostId pinnedHost : pinnedHosts) {
            HostTreeNode node = hostNodes.get(pinnedHost);
            if (node != null) {
                node.setPinned(true);
            }
        }

        int verifiedHostCount = verifiedHosts.size();
        int mainMirrorHostCount = topLevelNodes.size();

        return ImmutablePair.of(order(topLevelNodes), new HostCounters(unverifiedHostCount, verifiedHostCount, mainMirrorHostCount));
    }

    List<HostTreeNode> order(List<HostTreeNode> topLevelNodes) {
        Comparator<HostTreeNode> hostNameComparator = hostNameComparator();

        for (HostTreeNode node : topLevelNodes) {
            List<HostTreeNode> children = node.getChildren();
            if (!children.isEmpty()) {
                Collections.sort(children, hostNameComparator);
            }
        }

        ComparatorChain<HostTreeNode> comparatorChain = new ComparatorChain<>();
        comparatorChain.addComparator(pinnedComparator());
        comparatorChain.addComparator(hostNameComparator);

        Collections.sort(topLevelNodes, comparatorChain);
        return topLevelNodes;
    }

    public static class HostTreeNode {
        private final WebmasterHostId hostId;
        private boolean verified;
        private boolean pinned;
        private boolean added;

        private final List<HostTreeNode> children = new ArrayList<>();

        public HostTreeNode(WebmasterHostId hostId, boolean verified) {
            this.hostId = hostId;
            this.verified = verified;
            this.added = true;
        }

        public HostTreeNode(WebmasterHostId hostId) {
            this.hostId = hostId;
            this.verified = false;
            this.added = false;
        }

        public void setAdded(boolean added) {
            this.added = added;
        }

        public boolean isAdded() {
            return added;
        }

        void addChildren(HostTreeNode hostTreeNode) {
            children.add(hostTreeNode);
        }

        public WebmasterHostId getHostId() {
            return hostId;
        }

        public boolean isVerified() {
            return verified;
        }

        public void setVerified(boolean verified) {
            this.verified = verified;
        }

        public List<HostTreeNode> getChildren() {
            return children;
        }

        public boolean isPinned() {
            return pinned;
        }

        public void setPinned(boolean pinned) {
            this.pinned = pinned;
        }

        boolean hasPinnedNode() {
            if (isPinned()) {
                return true;
            }
            for (HostTreeNode child : children) {
                if (child.isPinned()) {
                    return true;
                }
            }
            return false;
        }

        public List<WebmasterHostId> getPinnedNodes() {
            List<WebmasterHostId> result = new ArrayList<>();
            if (isPinned()) {
                result.add(getHostId());
            }
            for (HostTreeNode child : children) {
                if (child.isPinned()) {
                    result.add(child.getHostId());
                }
            }
            return result;
        }
    }

    public static class HostCounters {
        private final int unverifiedHostsCount;
        private final int verifiedHostsCount;
        private final int mainMirrorHostsCount;

        public HostCounters(int unverifiedHostsCount, int verifiedHostsCount, int mainMirrorHostsCount) {
            this.unverifiedHostsCount = unverifiedHostsCount;
            this.verifiedHostsCount = verifiedHostsCount;
            this.mainMirrorHostsCount = mainMirrorHostsCount;
        }

        public int getUnverifiedHostsCount() {
            return unverifiedHostsCount;
        }

        public int getVerifiedHostsCount() {
            return verifiedHostsCount;
        }

        public int getMainMirrorHostsCount() {
            return mainMirrorHostsCount;
        }
    }

    public static Comparator<HostTreeNode> pinnedComparator() {
        return (node1, node2) -> {
            if (node1.hasPinnedNode()) {
                return node2.hasPinnedNode() ? 0 : -1;
            }
            return node2.hasPinnedNode() ? 1 : 0;
        };
    }

    public Comparator<HostTreeNode> hostNameComparator() {
        return (node1, node2) -> {
            HostDomainInfo host1 = hostDomainInfoService
                    .getDomainInfo(node1.getHostId().getPunycodeHostname());
            HostDomainInfo host2 = hostDomainInfoService
                    .getDomainInfo(node2.getHostId().getPunycodeHostname());
            return compareWebmasterHostId(node1.getHostId(), host1, node2.getHostId(), host2);
        };
    }

    public static int compareWebmasterHostId(WebmasterHostId hostId1, HostDomainInfo host1, WebmasterHostId hostId2, HostDomainInfo host2) {
        if (hostId1.equals(hostId2)) {
            return 0;
        }
        String prefix1 = StringUtils.trimToEmpty(host1.getPrefix());
        String middle1 = host1.getDomainMiddlePart();
        String owner1 = StringUtils.trimToEmpty(host1.getOwner());
        if (hostId1.isIDN()) {
            prefix1 = IdUtils.IDN.toUnicode(prefix1);
            middle1 = IdUtils.IDN.toUnicode(middle1);
            owner1 = IdUtils.IDN.toUnicode(owner1);
        }

        String prefix2 = StringUtils.trimToEmpty(host2.getPrefix());
        String middle2 = host2.getDomainMiddlePart();
        String owner2 = StringUtils.trimToEmpty(host2.getOwner());
        if (hostId2.isIDN()) {
            prefix2 = IdUtils.IDN.toUnicode(prefix2);
            middle2 = IdUtils.IDN.toUnicode(middle2);
            owner2 = IdUtils.IDN.toUnicode(owner2);
        }

        int midPartResult = compareDomain(middle1, middle2);
        if (midPartResult != 0) {
            return midPartResult;
        }

        int ownerResult = compareDomain(owner1, owner2);
        if (ownerResult != 0) {
            return ownerResult;
        }

        if (hostId1.getSchema() != hostId2.getSchema()) {
            return hostId1.getSchema() == WebmasterHostId.Schema.HTTP ? -1 : 1;
        }

        return prefix1.compareTo(prefix2);
    }

    private static int compareDomain(String domain1, String domain2) {
        String[] readableHostname1 = StringUtils.split(domain1, '.');
        ArrayUtils.reverse(readableHostname1);
        String[] readableHostname2 = StringUtils.split(domain2, '.');
        ArrayUtils.reverse(readableHostname2);
        int partsCount = Math.min(readableHostname1.length, readableHostname2.length);
        for (int i = 0; i < partsCount; i++) {
            String part1 = readableHostname1[i];
            String part2 = readableHostname2[i];

            int result = part1.compareTo(part2);
            if (result != 0) {
                return result;
            }
        }
        if (readableHostname1.length == readableHostname2.length) {
            return 0;
        }

        return readableHostname1.length < readableHostname2.length ? -1 : 1;
    }
}
