package ru.yandex.webmaster3.storage.util;

import java.io.IOException;
import java.io.InputStream;

import com.google.common.primitives.Ints;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.NullOutputStream;

/**
 * @author aherman
 */
public class LenvalStreamParser {
    private final InputStream is;
    private boolean streamEnded = false;

    public LenvalStreamParser(InputStream is) {
        this.is = is;
    }

    public void process(EntryProcessor entryProcessor) throws Exception {
        while (!streamEnded) {
            BoundedInputStream keyStream = getNextStream();
            if (keyStream == null) {
                streamEnded = true;
                break;
            }
            entryProcessor.processKey(keyStream);
            consumeStream(keyStream);

            BoundedInputStream valueStream = getNextStream();
            if (valueStream == null) {
                throw new LenvalFormatException("Value is empty");
            }
            entryProcessor.processValue(valueStream);
            consumeStream(valueStream);
        }
    }

    private BoundedInputStream getNextStream() throws LenvalFormatException, IOException {
        int i1 = is.read();
        if (i1 < 0) {
            return null;
        }
        int i2 = is.read();
        if (i2 < 0) {
            throw new LenvalFormatException("Unable to read next entry length, stream ended");
        }
        int i3 = is.read();
        if (i3 < 0) {
            throw new LenvalFormatException("Unable to read next entry length, stream ended");
        }
        int i4 = is.read();
        if (i4 < 0) {
            throw new LenvalFormatException("Unable to read next entry length, stream ended");
        }

        int length = Ints.fromBytes((byte) i4, (byte) i3, (byte) i2, (byte) i1);
        return new BoundedInputStream(this.is, length);
    }

    private void consumeStream(BoundedInputStream stream) throws IOException {
        if (stream.available() > 0) {
            IOUtils.copy(stream, new NullOutputStream());
        }
    }

    public interface EntryProcessor {
        public void processKey(InputStream is) throws Exception;
        public void processValue(InputStream is) throws Exception;
    }

    static class BoundedInputStream extends InputStream {
        private final int size;
        private int position;
        private final InputStream is;

        public BoundedInputStream(InputStream is, int size) {
            this.size = size;
            this.is = is;
        }

        @Override
        public int available() throws IOException {
            return size - position;
        }

        @Override
        public int read() throws IOException {
            if (position >= size) {
                return -1;
            }
            position++;
            return is.read();
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (position >= size) {
                return -1;
            }
            len = Math.min(len, size - position);
            int read = is.read(b, off, len);
            position += read;
            return read;
        }
    }
}
