package ru.yandex.webmaster3.storage.util;

import com.fasterxml.jackson.annotation.JsonProperty;
import ru.yandex.webmaster3.core.sitemap.SitemapErrorCode;
import ru.yandex.webmaster3.core.util.XmlUtil;
import ru.yandex.webmaster3.proto.Sitemap;
import ru.yandex.webmaster3.storage.sitemap.SitemapTypeEnum;

import java.util.LinkedList;
import java.util.List;

/**
 * Результат анализа файла sitemap
 *
 * User: azakharov
 * Date: 22.02.12
 * Time: 13:27
 */
public class SitemapAnalysisResult {
    private Integer urlCount;

    private List<SitemapErrorInfo> errorInfos;

    private Integer shownErrorsCount;
    private Integer totalErrorsCount;
    private Integer totalWarningsCount;

    private ru.yandex.webmaster3.storage.sitemap.SitemapTypeEnum sitemapType;

    private static final int MAX_SHOWN_ERRORS_COUNT = 100;

    public SitemapAnalysisResult(@JsonProperty("urlCount") Integer urlCount,
                                 @JsonProperty("errorInfos") List<SitemapErrorInfo> errorInfos,
                                 @JsonProperty("shownErrorsCount")Integer shownErrorsCount,
                                 @JsonProperty("totalErrorsCount")Integer totalErrorsCount,
                                 @JsonProperty("totalWarningsCount") Integer totalWarningsCount,
                                 @JsonProperty("sitemapType") SitemapTypeEnum sitemapType){
        this.urlCount = urlCount;
        this.errorInfos = errorInfos;
        this.shownErrorsCount = shownErrorsCount;
        this.totalErrorsCount = totalErrorsCount;
        this.totalWarningsCount = totalWarningsCount;
        this.sitemapType = sitemapType;
    }
    public SitemapAnalysisResult(Sitemap.SitemapInfo resultMessage) {
        urlCount = resultMessage.hasUrlCount() ? resultMessage.getUrlCount() : null;
        errorInfos = new LinkedList<>();
        totalErrorsCount = resultMessage.getErrorsList().size();
        shownErrorsCount = totalErrorsCount;
        if (totalErrorsCount > 0) {
            totalWarningsCount = 0;
            for (Sitemap.SitemapError info : resultMessage.getErrorsList()) {
                if (info.getCode().getNumber() < 100) {
                    totalWarningsCount++;
                }
            }

            final List<Sitemap.SitemapError> shownErrors = resultMessage.getErrorsList().subList(
                    0, Math.min(resultMessage.getErrorsList().size(), MAX_SHOWN_ERRORS_COUNT));
            shownErrorsCount = shownErrors.size();
            for (Sitemap.SitemapError info : shownErrors) {
                Integer lineNumber = info.hasLine() && info.getLine() > 0 ? info.getLine() : null;
                errorInfos.add(
                        new SitemapErrorInfo(
                                SitemapErrorCode.R.fromValue(info.getCode().getNumber()),
                                lineNumber,
                                info.hasText() ? filterBinaryMessage(info.getText()) : ""));
            }
        }
        if (resultMessage.hasTypeEx()) {
            sitemapType = ru.yandex.webmaster3.storage.sitemap.SitemapTypeEnum.R.fromValueOrNull(resultMessage.getTypeEx().getNumber());
        }
    }

    public String filterBinaryMessage(String data) {
        return XmlUtil.isValidXmlString(data) ? data : "";
    }

    public Integer getUrlCount() {
        return urlCount;
    }

    public List<SitemapErrorInfo> getErrorInfos() {
        return errorInfos;
    }

    public void addWrnRedirected() {
        // Если ошибок много, то заменяем одну из показываемых на
        if (shownErrorsCount == MAX_SHOWN_ERRORS_COUNT) {
            errorInfos.remove(errorInfos.size() - 1);
        } else {
            shownErrorsCount++;
        }
        errorInfos.add(new SitemapErrorInfo(SitemapErrorCode.WRN_SITEMAP_URL_REDIRECTED, null, ""));
        totalErrorsCount++;
        totalWarningsCount++;
    }

    public Integer getShownErrorsCount() {
        return shownErrorsCount;
    }

    public Integer getTotalErrorsCount() {
        return totalErrorsCount;
    }

    public Integer getTotalWarningsCount() {
        return totalWarningsCount;
    }

    public ru.yandex.webmaster3.storage.sitemap.SitemapTypeEnum getSitemapType() {
        return sitemapType;
    }

    public static class SitemapErrorInfo {
        SitemapErrorCode code;
        Integer lineNumber;
        String message;

        public SitemapErrorInfo(SitemapErrorCode code, Integer lineNumber, String message) {
            this.code = code;
            this.lineNumber = lineNumber;
            this.message = message;
        }

        public SitemapErrorCode getCode() {
            return code;
        }

        public Integer getLineNumber() {
            return lineNumber;
        }

        public String getMessage() {
            return message;
        }
    }

}
