package ru.yandex.webmaster3.storage.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import wmc.stubs.Stubs;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.robotstxt.AllowInfo;
import ru.yandex.webmaster3.core.robotstxt.AnalysisResult;
import ru.yandex.webmaster3.core.robotstxt.ErrorInfo;
import ru.yandex.webmaster3.core.robotstxt.FormatErrorType;
import ru.yandex.webmaster3.proto.Sitemap;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.LightDispatcherHttpService;

/**
 * @author aherman
 */
public class W3DispatcherHttpService extends LightDispatcherHttpService{
    private static final Logger log = LoggerFactory.getLogger(W3DispatcherHttpService.class);

    public static final String SITEMAP_REQUEST_DATA_PARAM = "data=";

    public AnalysisResult analyzeRobotsTxt(final String robotsTxt, final List<String> urls) throws WebmasterException {
        Stubs.analysis_result_msg.Builder ans;
        try {
            InputStream is = httpPostResponse("analyzerobotstxt",
                    createParams()
                            .addParam("robotstxt", robotsTxt)
                            .addParam("urls", String.join("\n", urls)),
                    SOCKET_TIMEOUT);

            ans = Stubs.analysis_result_msg.newBuilder().mergeFrom(is);
        } catch (Exception e) {
            throw new WebmasterException("Unable to analyze robots.txt",
                    new WebmasterErrorResponse.DispatcherErrorResponse(this.getClass(), e), e);
        }
        return createAnalysisResult(ans.build());
    }

    private static AnalysisResult createAnalysisResult(Stubs.analysis_result_msg msg) {
        return new AnalysisResult(getErrors(msg), msg.getAcceptedLinesList(), getAllowInfo(msg));
    }

    private static List<ErrorInfo> getErrors(Stubs.analysis_result_msg msg) {
        return msg.getErrorsList().stream()
                .map((oldInfo) -> {
                    FormatErrorType type = FormatErrorType.R.fromValueOrNull(oldInfo.getType().getNumber());
                    return new ErrorInfo(type, (int) oldInfo.getLineNum());
                })
                .collect(Collectors.toList());
    }

    private static List<AllowInfo> getAllowInfo(Stubs.analysis_result_msg msg) {
        return msg.getAreAllowedList().stream()
                .map(allowInfo -> {
                    boolean syntaxError = allowInfo.hasType() && allowInfo.getType() == Stubs.url_error_type.ERR_URL_SYNTAX;
                    String changedUrl = allowInfo.hasChangedUrl() ? allowInfo.getChangedUrl() : null;

                    return new AllowInfo(allowInfo.getAllowed(), allowInfo.getRule(), syntaxError, changedUrl);
                })
                .collect(Collectors.toList());
    }



    public SitemapAnalysisResult analyzeSitemapBytes(byte[] data)
            throws InternalException, UserException, UnsupportedEncodingException
    {
        checkSitemapDataPrefix(data);
        log.debug(URLEncoder.encode(new String(
                Arrays.copyOfRange(data, SITEMAP_REQUEST_DATA_PARAM.length(), Math.min(data.length, 256))), "UTF-8"));

        final Sitemap.SitemapInfo ans;
        try {
            InputStream result = httpPostByteArrayResponse("analyzeSitemap", data, SOCKET_TIMEOUT, true);
            ans = Sitemap.SitemapInfo.parseFrom(result);
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
        return new SitemapAnalysisResult(ans);
    }

    private void checkSitemapDataPrefix(byte[] data) {
        byte[] prefix = SITEMAP_REQUEST_DATA_PARAM.getBytes();
        if (data.length < prefix.length) {
            throw new IllegalArgumentException("Sitemap data not prefixed");
        }
        for (int i = 0; i < prefix.length; i++) {
            if (prefix[i] != data[i]) {
                throw new IllegalArgumentException("Sitemap data not prefixed");
            }
        }
    }

}
