package ru.yandex.webmaster3.storage.util.clickhouse2;

import com.google.common.hash.Hashing;
import lombok.Value;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.base.AbstractInstant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.CityHash102;
import ru.yandex.webmaster3.core.util.FNVHash;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.util.UUIDUtil;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.From;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Statement;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.cases.Case;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;

/**
 * @author aherman
 */
public abstract class AbstractClickhouseDao {
    private static final Logger log = LoggerFactory.getLogger(AbstractClickhouseDao.class);
    public static String DB_WEBMASTER3 = "webmaster3";
    public static String DB_WEBMASTER3_INDEXING = "webmaster3_indexing";
    public static String DB_WEBMASTER3_QUERIES = "webmaster3_queries";
    public static String DB_WEBMASTER3_SEARCHURLS = "webmaster3_searchurls";
    public static String DB_WEBMASTER3_LINKS = "webmaster3_links";
    public static String DB_WEBMASTER3_NOTIFICATIONS = "webmaster3_notifications";
    public static String DB_WEBMASTER3_TURBO = "webmaster3_turbo";
    public static String DB_WEBMASTER3_IMPORTANTURLS = "webmaster3_importanturls";
    public static String DB_WEBMASTER3_CHECKLIST = "webmaster3_checklist";
    public static String DB_WEBMASTER3_NICHE = "webmaster3_niche";
    public static String DB_WEBMASTER3_FEEDS = "webmaster3_feeds";
    public static String DB_WEBMASTER3_NICHE2 = "webmaster3_niche2";

    protected static final String CREATE_TABLE_TEMPLATE_PREFIX = "CREATE TABLE %1s.%2s ";
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");
    private static final DateTimeFormatter DATE_TIME_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    public static final String UUID_MS_POSTFIX = "_ms";
    public static final String UUID_LS_POSTFIX = "_ls";

    private ClickhouseServer clickhouseServer;

    protected <T, A> T collectAll(String query, Collector<CHRow, A, T> collector) throws ClickhouseException {
        return getClickhouseServer().collectAll(query, collector);
    }

    protected <T, A> T collectAll(String query, InputStream data, Collector<CHRow, A, T> collector) throws ClickhouseException {
        return getClickhouseServer().collectAll(query, data, collector);
    }

    protected <T, A> T collectAll(Statement st, Collector<CHRow, A, T> collector) throws ClickhouseException {
        return getClickhouseServer().collectAll(st.toString(), collector);
    }

    protected void forEach(String query, Consumer<CHRow> consumer) throws ClickhouseException {
        getClickhouseServer().forEach(query, consumer);
    }

    protected void forEach(String query, InputStream data, Consumer<CHRow> consumer) throws ClickhouseException {
        getClickhouseServer().forEach(query, data, consumer);
    }

    protected void forEach(Statement st, Consumer<CHRow> consumer) throws ClickhouseException {
        getClickhouseServer().forEach(st.toString(), consumer);
    }

    protected <T> List<T> queryAll(String query, Function<CHRow, T> mapper) throws ClickhouseException {
        return getClickhouseServer().selectAll(query, mapper);
    }

    protected <T> List<T> queryAll(Statement st, Function<CHRow, T> mapper) throws ClickhouseException {
        return queryAll(st.toString(), mapper);
    }

    protected <T> Optional<T> queryOne(String query, Function<CHRow, T> mapper) throws ClickhouseException {
        return getClickhouseServer().selectOne(query, mapper);
    }

    protected <T> Optional<T> queryOne(Statement st, Function<CHRow, T> mapper) throws ClickhouseException {
        return queryOne(st.toString(), mapper);
    }

    protected void insert(String query) throws ClickhouseException {
        getClickhouseServer().insert(query);
    }

    protected void insert(Statement st) throws ClickhouseException {
        insert(st.toString());
    }

    protected void insert(String query, InputStream stream) throws ClickhouseException {
        getClickhouseServer().insert(query, stream);
    }

    protected void insert(Statement st, InputStream stream) throws ClickhouseException {
        insert(st.toString(), stream);
    }

    protected void executeOnAllHosts(String query) throws ClickhouseException {
        getClickhouseServer().executeOnAllHosts(ClickhouseQueryContext.useDefaults(), query);
    }

    protected <T> T executeWithFixedHost(ClickhouseServer.TxCallback<T> tx) throws ClickhouseException {
        return getClickhouseServer().executeWithFixedHost(tx);
    }

    protected static WebmasterHostId getWebmasterHostId(CHRow r, String fieldName) {
        String id = r.getString(fieldName);
        return IdUtils.stringToHostId(id);
    }

    protected byte[] uuidToBytes(UUID uuid) {
        return UUIDUtil.uuidToBytes(uuid);
    }

    protected UUID getUUIDFromBytes(byte[] bytes) {
        return UUIDUtil.bytesToUUID(bytes);
    }

    protected static UUID getUUID(CHRow r, String name) {
        return new UUID(
                r.getLong(name + UUID_MS_POSTFIX),
                r.getLong(name + UUID_LS_POSTFIX)
        );
    }

    /**
     * @deprecated теряет TimeZone. Лучше вообще хранить время timestamp'ом
     */
    @Deprecated
    public static String toClickhouseDate(Instant dateTime) {
        return dateTime.toString(DATE_FORMAT);
    }

    /**
     * @deprecated теряет TimeZone. Лучше вообще хранить время timestamp'ом
     */
    @Deprecated
    public static String toClickhouseDate(DateTime dateTime) {
        return dateTime.toString(DATE_FORMAT);
    }

    public static String toClickhouseDate(LocalDate date) {
        return date.toString(DATE_FORMAT);
    }

    /**
     * @deprecated теряет TimeZone. Лучше вообще хранить время timestamp'ом
     */
    @Deprecated
    public static String toClickhouseDateTime(AbstractInstant dateTime) {
        return dateTime.toString(DATE_TIME_FORMAT);
    }

    protected static DateTime getDate(CHRow r, String fieldName) {
        return DateTime.parse(r.getString(fieldName), DATE_FORMAT);
    }

    protected static LocalDate getLocalDate(CHRow r, String fieldName) {
        return LocalDate.parse(r.getString(fieldName), DATE_FORMAT);
    }

    protected static DateTime getDateTime(CHRow r, String fieldName) {
        return DateTime.parse(r.getString(fieldName), DATE_TIME_FORMAT);
    }

    public static long toDateTimeAsTimestamp(DateTime dateTime) {
        return dateTime.getMillis();
    }

    protected InputStream packValues(Object... values) {
        SimpleByteArrayOutputStream bs = new SimpleByteArrayOutputStream();
        return packRowValues(bs, values).toInputStream();
    }

    protected SimpleByteArrayOutputStream packRowValues(SimpleByteArrayOutputStream bs, Object... values) {
        BAOutputStream escaped = ClickhouseEscapeUtils.escape(bs);

        for (int i = 0; i < values.length; i++) {
            Object v = values[i];
            if (v instanceof WebmasterHostId) {
                escaped.write(((WebmasterHostId) v).toStringId().getBytes(StandardCharsets.UTF_8));
            } else if (v instanceof byte[]) {
                escaped.write((byte[]) v);
            } else if (v instanceof Set && (((Set) v).size() == 0 || ((Set) v).iterator().next() instanceof String)) {
                // эскейпим здесь, а не в ClickhouseEscapeUtils, поэтому сразу пишем в bs
                bs.write(toClickhouseStringArray((Set<String>)v).getBytes(StandardCharsets.UTF_8));
            } else {
                escaped.write(v.toString().getBytes(StandardCharsets.UTF_8));
            }
            if (i < values.length - 1) {
                bs.write('\t');
            }
        }
        bs.write('\n');

        return bs;
    }

    public static long sipHash24(String v) {
        return Hashing.sipHash24(0, 0).hashUnencodedChars(v).asLong();
    }

    public Where whereOrFrom(Where where, From from, Case case_) {
        if (where != null) {
            return where.and(case_);
        } else if (from != null) {
            return from.where(case_);
        }
        return null;
    }

    protected static int booleanToInt(boolean value) {
        return value ? 1 : 0;
    }

    protected static WebmasterHostId getNullableHostId(CHRow row, String field) {
        String result = row.getString(field);
        if (StringUtils.isEmpty(result)) {
            return null;
        } else {
            return IdUtils.stringToHostId(result);
        }
    }

    protected static DateTime getNullableDate(CHRow row, String field) {
        long result = row.getLong(field);
        if (result == 0L) {
            return null;
        } else {
            return new DateTime(result * 1000);
        }
    }

    protected ClickhouseQueryContext.Builder withShard(int shard) throws ClickhouseException {
        return ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostOrFail(shard));
    }

    protected ClickhouseQueryContext.Builder withMaxShard(int shard) throws ClickhouseException {
        return ClickhouseQueryContext.useDefaults().setHost(getClickhouseServer().pickAliveHostWithMaxShardOrFail(shard));
    }

    protected static String toClickhouseStringArray(Set<String> set) {
        return Arrays.toString(set.stream().filter(Objects::nonNull).map(AbstractClickhouseDao::toClickhouseStringAsArrayElement).toArray());
    }

    public static String toClickhouseStringAsArrayElement(String value) {
        return "'" + ClickhouseEscapeUtils.escapeString(value) + "'";
    }

    protected static String toStringOrEmpty(Object o) {
        if (o == null) {
            return "";
        } else {
            return o.toString();
        }
    }

    protected static long zeroOnNull(Long v) {
        return v == null ? 0L : v;
    }

    protected static Set<String> emptyOnNullAndRemoveNullItems(Set<String> set) {
        return set == null ? Collections.EMPTY_SET : set.stream().filter(Objects::nonNull).collect(Collectors.toSet());
    }

    protected static String getNullableString(CHRow row, String field) {
        String res = row.getString(field);
        return res.isEmpty() ? null : res;
    }

    protected static Long getNullableLong(CHRow row, String field) {
        long result = row.getLongUnsafe(field);
        return result == 0 ? null : result;
    }

    @Required
    public void setClickhouseServer(ClickhouseServer clickhouseServer) {
        this.clickhouseServer = clickhouseServer;
    }

    protected ClickhouseServer getClickhouseServer() {
        return clickhouseServer;
    }

    @Value(staticConstructor = "of")
    public static final class BuilderMapper<Builder, Field> {
        String name;
        BiFunction<CHRow, String, Field> fieldMapper;
        BiFunction<Builder, Field, Builder> fieldBuilder;

        public void apply(CHRow row, Builder builder) {
            fieldBuilder.apply(builder, fieldMapper.apply(row, name));
        }
    }

    protected int getShard(String domain) {
        return (int) FNVHash.hash64Mod(domain, getClickhouseServer().getShardsCount());
    }

    protected ClickhouseQueryContext.Builder chContext(String domain) {
        int shard = getShard(domain);
        List<ClickhouseHost> aliveHosts = getClickhouseServer().getHosts().stream().filter(ClickhouseHost::isUp)
                .filter(host -> host.getShard() == shard).collect(Collectors.toList());
        if (aliveHosts.isEmpty()) {
            throw new ClickhouseException("Not found alive host for shard " + shard, null, null);
        }
        // choose random DC, but prefer same
        int dc = (int) Long.remainderUnsigned(CityHash102.cityHash64(String.valueOf(domain)), aliveHosts.size());
        return ClickhouseQueryContext.useDefaults().setHost(aliveHosts.get(dc));
    }

}
