package ru.yandex.webmaster3.storage.util.clickhouse2;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Getter;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableSource;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;

/**
 * Created by Oleg Bazdyrev on 21/04/2017.
 */
public class CHTable {

    public static final String MERGE_TABLE_SUFFIX = "_merge";
    public static final String DISTRIB_TABLE_SUFFIX = "_distrib";
    // исторически сложилось, что куски называются shard, хотя логичнее было бы part
    private final String database;
    @Getter
    private final String name;
    private final String partNameSuffix;
    private final List<CHField> fields;
    private final List<CHField> keyFields;
    private final CHField keyDateField;
    private final int parts;
    @Getter
    private final boolean sharded;
    private final String partitionBy;
    private final Map<String, Object> settings;

    private CHTable(String database, String name, String partNameSuffix, List<CHField> fields, List<CHField> keyFields,
                    CHField keyDateField, int parts, boolean sharded, String partitionBy, Map<String, Object> settings) {
        this.database = database;
        this.name = name;
        this.partNameSuffix = partNameSuffix;
        this.fields = fields;
        this.keyFields = keyFields;
        this.keyDateField = keyDateField;
        this.parts = parts;
        this.settings = settings == null ? new HashMap<>() : settings;
        this.sharded = sharded;
        this.partitionBy = partitionBy;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(database, name, partNameSuffix, fields, keyFields, keyDateField, parts, sharded, partitionBy, settings);
    }

    public String createSpec(String nameSuffix, boolean addCreateTable, Object... params) {
        return createSpec(nameSuffix, addCreateTable, false, params);
    }

    public String createSpec(String nameSuffix, boolean addCreateTable, boolean onCluster, Object... params) {
        StringBuilder sb = new StringBuilder(1024);
        if (addCreateTable) {
            sb.append("CREATE TABLE IF NOT EXISTS ").append(database).append(".").append(String.format(name, params))
                    .append(nameSuffix);
            if (onCluster) {
                sb.append(" ON CLUSTER '{cluster}'");
            }
        }
        sb.append(" (");
        // можно и через Collectors.joining, только слишком много строк будет создаваться
        boolean first = true;
        for (CHField field : fields) {
            if (first) {
                first = false;
            } else {
                sb.append(", ");
            }
            sb.append(field.getName()).append(" ").append(field.getType());
        }
        sb.append(")");
        return sb.toString();
    }

    private String getGranularity() {
        return settings.getOrDefault("index_granularity", 8192).toString();
    }

    private String replicatedMergeTreeEngine(String nameSuffix, Object... params) {
        Preconditions.checkState(keyDateField != null || partitionBy != null);
        Preconditions.checkState(keyFields != null && !keyFields.isEmpty());
        String keyFieldsString = keyFields.stream().map(CHField::getName).collect(Collectors.joining(",", "(", ")"));
        if (keyDateField != null) {
            return " ENGINE = ReplicatedMergeTree('/webmaster3/clickhouse/tables/" + database + "/" +
                    String.format(name, params) + nameSuffix + (sharded ? "/{shard}" : "") + "', '{replica}', " + keyDateField.getName() + ", " +
                    keyFieldsString + ", " + getGranularity() + ")";
        } else {
            StringBuilder result = new StringBuilder(256);
            result.append(" ENGINE = ReplicatedMergeTree('/webmaster3/clickhouse/tables/").append(database).append("/");
            result.append(String.format(name, params)).append(nameSuffix).append(sharded ? "/{shard}" : "").append("', '{replica}')");
            if (!partitionBy.isEmpty()) {
                result.append(" PARTITION BY ").append(partitionBy);
            }
            result.append(" ORDER BY ").append(keyFieldsString);
            if (!settings.isEmpty()) {
                result.append(" SETTINGS ");
                result.append(settings.entrySet().stream().map(entry -> entry.getKey() + " = " + entry.getValue()).collect(Collectors.joining(", ")));
            }
            return result.toString();
        }
    }

    private String mergeTreeEngine(String nameSuffix, Object... params) {
        Preconditions.checkState(keyDateField != null || partitionBy != null);
        Preconditions.checkState(keyFields != null && !keyFields.isEmpty());
        String keyFieldsString = keyFields.stream().map(CHField::getName).collect(Collectors.joining(",", "(", ")"));
        if (keyDateField != null) {
            return " ENGINE = MergeTree(" + keyDateField.getName() + ", " + keyFieldsString + ", " + getGranularity() + ")";
        } else {
            StringBuilder result = new StringBuilder(256);
            result.append("ENGINE = MergeTree() ");
            if (!partitionBy.isEmpty()) {
                result.append(" PARTITION BY ").append(partitionBy);
            }
            result.append(" ORDER BY ").append(keyFieldsString);
            if (!settings.isEmpty()) {
                result.append(" SETTINGS ");
                result.append(settings.entrySet().stream().map(entry -> entry.getKey() + " = " + entry.getValue()).collect(Collectors.joining(", ")));
            }
            return result.toString();
        }
    }

    private String mergeEngine(Object... params) {
        StringBuilder sb = new StringBuilder(128);
        sb.append(" ENGINE = Merge('").append(database).append("', '^").append(String.format(name, params));
        if (!Strings.isNullOrEmpty(partNameSuffix)) {
            sb.append(partNameSuffix);
        }
        sb.append("')");
        return sb.toString();
    }

    private String distributedEngine(String cluster, boolean onMergeTable, Object... params) {
        StringBuilder sb = new StringBuilder(128);
        sb.append(" ENGINE = Distributed(" + cluster + ", '").append(database).append("', '").append(String.format(name, params));
        if (onMergeTable) {
            sb.append(MERGE_TABLE_SUFFIX);
        }
        sb.append("')");
        return sb.toString();
    }

    /**
     * Create statement without CREATE TABLE <table-name>
     */
    public String createReplicatedMergeTreeSpec(int part, Object... params) {
        String suffix = partSuffix(part);
        return createSpec(suffix, false, params) + replicatedMergeTreeEngine(suffix, params);
    }

    public String createMergeTreeSpec(int part, Object... params) {
        String suffix = partSuffix(part);
        return createSpec(suffix, false, params) + mergeTreeEngine(suffix, params);
    }

    @NotNull
    private String partSuffix(int part) {
        return (part >= 0 && !Strings.isNullOrEmpty(partNameSuffix)) ? (partNameSuffix + part) : "";
    }

    public String getPartNameSuffix() {
        return partNameSuffix;
    }

    /**
     * Create statement with CREATE TABLE <table-name>
     */
    public String createReplicatedMergeTree(int part, Object... params) {
        String suffix = partSuffix(part);
        return createSpec(suffix, true, params) + replicatedMergeTreeEngine(suffix, params);
    }

    public String createMerge(Object... params) {
        return createSpec(MERGE_TABLE_SUFFIX, true, params) + mergeEngine(params);
    }

    public String createDistributed(Object... params) {
        return createSpec(DISTRIB_TABLE_SUFFIX, true, params) + distributedEngine("'{cluster}'", true, params);
    }

    public String createDistributedSymlink(Object... params) {
        String[] empty = Arrays.stream(params).map(i -> "").toArray(String[]::new);
        String createSpec = createSpec(DISTRIB_TABLE_SUFFIX, true, true, empty).replaceAll("_{2,}", "_");
        if (sharded) {
            return createSpec + distributedEngine("'{cluster}'", parts > 1, params);
        } else {
            return createSpec + mergeEngine(params);
        }
    }

    public String dropDistributedSymlink() {
        StringBuilder sb = new StringBuilder(1024);
        sb.append("DROP TABLE IF EXISTS ").append(database).append(".").append(String.format(name, "", "", "", ""))
                .append(DISTRIB_TABLE_SUFFIX).append(" ON CLUSTER '{cluster}'");
        return sb.toString().replaceAll("_{2,}", "_");
    }

    public void updateDistributedSymlink(ClickhouseServer clickhouseServer, Object... params) {
        clickhouseServer.execute(dropDistributedSymlink());
        clickhouseServer.execute(createDistributedSymlink(params));
    }

    public String insertSpec(int part, Object... params) {
        StringBuilder sb = new StringBuilder("INSERT INTO ");
        sb.append(database).append(".").append(String.format(name, params)).append(partSuffix(part));
        sb.append(importSpec());
        return sb.toString();
    }

    public String importSpec() {
        return fields.stream().map(CHField::getName).collect(Collectors.joining(", ", "(", ") FORMAT TabSeparated"));
    }

    public String replicatedMergeTreeTableName(int part, Object... params) {
        String suffix = partSuffix(part);
        return String.format(name, params) + suffix;
    }

    public String mergeTableName(Object... params) {
        return String.format(name, params) + MERGE_TABLE_SUFFIX;
    }

    public String distributedTableName(Object... params) {
        return String.format(name, params) + DISTRIB_TABLE_SUFFIX;
    }

    public ClickhouseTableInfo toClickhouseTableInfo(TableType type, int shards, Object... params) {
        return new ClickhouseTableInfo(type, UUIDs.timeBased(),
                TableState.DEPLOYED, DateTime.now(), TableSource.YT_HAHN, null,
                database + "." + distributedTableName(params),
                database + "." + distributedTableName(params),
                database + "." + mergeTableName(params), shards, parts);
    }

    public ClickhouseTableInfo toClickhouseTableInfoWithoutShards(TableType type, Object... params) {
        return new ClickhouseTableInfo(type, UUIDs.timeBased(),
                TableState.DEPLOYED, DateTime.now(), TableSource.YT_HAHN, null,
                database + "." + distributedTableName(params),
                database + "." + distributedTableName(params),
                database + "." + String.format(name, params), 1, 1);
    }

    public String getDatabase() {
        return database;
    }

    public List<CHField> getFields() {
        return fields;
    }

    public int getParts() {
        return parts;
    }

    public static class Builder {

        private String database;
        private String name;
        private String partNameSuffix;
        private List<CHField> fields = new ArrayList<>();
        private List<CHField> keyFields = new ArrayList<>();
        private CHField keyDateField;
        private int parts = 1;
        private boolean sharded = true;
        private String partitionBy = null;
        private Map<String, Object> settings = new HashMap<>();

        private Builder() {

        }

        public Builder(String database, String name, String partNameSuffix, List<CHField> fields, List<CHField> keyFields, CHField keyDateField,
                       int parts, boolean sharded, String partitionBy, Map<String, Object> settings) {
            this.database = database;
            this.name = name;
            this.partNameSuffix = partNameSuffix;
            this.fields = fields;
            this.keyFields = keyFields;
            this.keyDateField = keyDateField;
            this.parts = parts;
            this.sharded = sharded;
            this.partitionBy = partitionBy;
            this.settings = settings == null ? new HashMap<>() : settings;
        }

        public Builder database(String database) {
            this.database = database;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder partNameSuffix(String partNameSuffix) {
            this.partNameSuffix = partNameSuffix;
            return this;
        }

        public Builder field(String name, CHType type) {
            fields.add(new CHField(name, type));
            return this;
        }

        public Builder keyField(String name, CHType type) {
            CHField field = new CHField(name, type);
            fields.add(field);
            keyFields.add(field);
            return this;
        }

        public Builder keyDateField(String name) {
            Preconditions.checkState(keyDateField == null);
            keyDateField = new CHField(name, CHPrimitiveType.Date);
            fields.add(keyDateField);
            partitionBy = null;
            return this;
        }

        public Builder parts(int parts) {
            this.parts = parts;
            return this;
        }

        public Builder addSetting(String key, Object value) {
            this.settings.put(key, value);
            return this;
        }

        public Builder partsToThrowInsert(int value) {
            return addSetting("parts_to_throw_insert", value);
        }

        public Builder sharded(boolean sharded) {
            this.sharded = sharded;
            return this;
        }

        public Builder partitionBy(String partitionBy) {
            this.partitionBy = partitionBy;
            this.keyDateField = null;
            return this;
        }

        public Builder keyFields(String... names) {
            this.keyFields = new ArrayList<>();
            for (String name : names) {
                this.keyFields.add(fields.stream().filter(f -> f.getName().equals(name)).findFirst().orElseThrow());
            }
            return this;
        }

        public CHTable build() {
            Preconditions.checkState(!Strings.isNullOrEmpty(database));
            Preconditions.checkState(!Strings.isNullOrEmpty(name));
            Preconditions.checkState(!fields.isEmpty());
            return new CHTable(database, name, partNameSuffix, fields, keyFields, keyDateField, parts, sharded, partitionBy, settings);
        }

    }

}
