package ru.yandex.webmaster3.storage.util.clickhouse2;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.http.auth.UsernamePasswordCredentials;
import org.joda.time.Duration;

/**
 * @author aherman
 */
public class ClickhouseQueryContext {
    private final Duration timeout;
    private final ClickhouseHost host;
    private final UsernamePasswordCredentials credentials;
    private final boolean logQuery;
    private final Map<String, String> settings;

    private ClickhouseQueryContext(Duration timeout, ClickhouseHost host,
                                   UsernamePasswordCredentials credentials, boolean logQuery, Map<String, String> settings)
    {
        this.timeout = timeout;
        this.host = host;
        this.credentials = credentials;
        this.logQuery = logQuery;
        this.settings = settings;
    }

    public Duration getTimeout() {
        return timeout;
    }

    public ClickhouseHost getHost() {
        return host;
    }

    public UsernamePasswordCredentials getCredentials() {
        return credentials;
    }

    public boolean isLogQuery() {
        return logQuery;
    }

    public Map<String, String> getSettings() {
        return settings;
    }

    public static Builder useDefaults() {
        return new Builder(Optional.empty(), Optional.empty(), Optional.empty(), true, new HashMap<>());
    }

    public static class Builder {
        private final Optional<Duration> timeout;
        private final Optional<ClickhouseHost> host;
        private final Optional<UsernamePasswordCredentials> credentials;
        private final boolean logQuery;
        private final Map<String, String> settings;

        private Builder(Optional<Duration> timeout, Optional<ClickhouseHost> host, Optional<UsernamePasswordCredentials> credentials,
                        boolean logQuery, Map<String, String> settings)
        {
            this.timeout = timeout;
            this.host = host;
            this.credentials = credentials;
            this.logQuery = logQuery;
            this.settings = settings;
        }

        public Builder setTimeout(Duration timeout) {
            return new Builder(Optional.of(timeout), this.host, this.credentials, this.logQuery, settings);
        }

        public Builder setHost(ClickhouseHost clickhouseHost) {
            return new Builder(this.timeout, Optional.of(clickhouseHost), this.credentials, this.logQuery, settings);
        }

        public Builder setCredentials(UsernamePasswordCredentials credentials) {
            return new Builder(this.timeout, this.host, Optional.of(credentials), this.logQuery, settings);
        }

        public Builder setLogQuery(boolean doLog) {
            return new Builder(this.timeout, this.host, this.credentials, doLog, settings);
        }

        public Builder addParameter(String name, String value) {
            settings.put(name, value);
            return this;
        }

        Optional<Duration> getTimeout() {
            return timeout;
        }

        Optional<ClickhouseHost> getHost() {
            return host;
        }

        Optional<UsernamePasswordCredentials> getCredentials() {
            return credentials;
        }

        ClickhouseQueryContext build() {
            return new ClickhouseQueryContext(timeout.get(), host.get(), credentials.get(), logQuery, settings);
        }
    }
}
