package ru.yandex.webmaster3.storage.util.clickhouse2;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Iterator;

/**
 * @author aherman
 */
public class LineInputStream implements Iterable<LineInputStream.Line> {
    private final InputStream is;
    private byte[] buffer;
    private int lineStart = 0;
    private int lineEnd = 0;
    private int end = 0;
    private boolean streamEnded = false;

    public LineInputStream(InputStream is, int bufferSize) {
        this.is = is;
        this.buffer = new byte[bufferSize];
    }

    private void refill() throws IOException {
        if (streamEnded) {
            return;
        }

        int len = end - lineStart;
        if (len > 0) {
            System.arraycopy(buffer, lineStart, buffer, 0, len);
            end = len;
        } else {
            end = 0;
        }
        lineStart = 0;

        while (!streamEnded && buffer.length - end > 0) {
            int read = holdLine();
            if (read > 0) {
                end += read;
            } else {
                streamEnded = true;
            }
        }
    }

    private int holdLine() throws IOException {
        int read = is.read(buffer, end, buffer.length - end);
        if (read == 0) {
            return 0;
        }
        while (!stopLine(read)) {
            end += read;
            int len = buffer.length;
            int newSize = (len * 3) / 2 + 1;
            byte[] newBuffer = new byte[newSize];
            System.arraycopy(buffer, 0, newBuffer, 0, len);
            buffer = newBuffer;
            read = is.read(buffer, len, newSize - len);
        }
        return read;
    }

    private boolean stopLine(int read) {
        return (ArrayUtils.contains(buffer, (byte) '\n') || end + read < buffer.length);
    }

    private boolean findNextLine() throws IOException {
        if (lineEnd + 1 < end) {
            int nextLineStart = lineEnd + 1;
            int nextLineEnd = ArrayUtils.indexOf(buffer, (byte) '\n', nextLineStart);
            if (nextLineEnd >= 0 && nextLineEnd < end) {
                lineStart = nextLineStart;
                lineEnd = nextLineEnd;

                return true;
            }
        }

        lineStart = lineEnd + 1;
        refill();
        lineEnd = ArrayUtils.indexOf(buffer, (byte) '\n', lineStart);
        if (lineEnd > end || lineEnd < 0) {
            if (streamEnded) {
                lineEnd = end;
            } else {
                throw new IllegalStateException("Line buffer are too small to hold line");
            }
        }
        return lineStart < lineEnd;
    }

    @Override
    public Iterator<Line> iterator() {
        return new LBIterator();
    }

    private class LBIterator implements Iterator<Line> {
        @Override
        public boolean hasNext() {
            try {
                findNextLine();
            } catch (IOException e) {
                throw new IllegalStateException("Error in stream", e);
            }
            return lineStart < end;
        }

        @Override
        public Line next() {
            return new Line(lineStart, lineEnd);
        }
    }

    public class Line {
        private final int start;
        private final int end;

        public Line(int start, int end) {
            this.start = start;
            this.end = end;
        }

        private void checkIndex(int idx) {
            if (start + idx > end) {
                throw new IllegalArgumentException();
            }
        }

        private void checkRange(int from, int to) {
            checkIndex(from);
            checkIndex(to);
            if (from > to) {
                throw new IllegalArgumentException("Strange range [" + from + ", " + to + ")");
            }
        }

        public BAInputStream getBytes(int from) {
            checkIndex(from);
            if (from == end - start) {
                return new BAInputStream.EmptyBAInputStream();
            }
            return new BufferInputStream(start + from, end);
        }

        public BAInputStream getBytes(int from, int to) {
            checkRange(from, to);
            if (from == to) {
                return new BAInputStream.EmptyBAInputStream();
            }
            return new BufferInputStream(start + from, start + to);
        }

        public String getString(int from, Charset charset) {
            checkIndex(from);
            if (start + from == end) {
                return StringUtils.EMPTY;
            }
            return new String(buffer, start + from, end - start - from, charset);
        }

        public String getString(int from, int to, Charset charset) {
            checkRange(from, to);
            if (from == to) {
                return StringUtils.EMPTY;
            }
            return new String(buffer, start + from, to - from, charset);
        }

        public InputStream getInputStream(int from, int to) {
            checkRange(from, to);
            return new ByteArrayInputStream(buffer, start + from, to - from);
        }

        public int indexOf(byte b) {
            for (int i = start; i < end; i++) {
                if (buffer[i] == b) {
                    return i - start;
                }
            }

            return -1;
        }

        public int indexOf(byte b, int from) {
            checkIndex(from);
            if (from == end - start) {
                return -1;
            }
            for (int i = start + from; i < end; i++) {
                if (buffer[i] == b) {
                    return i - start;
                }
            }

            return -1;
        }

        public int count(byte b) {
            int count = 0;
            for (int i = start; i < end; i++) {
                if (buffer[i] == b) {
                    count++;
                }
            }

            return count;
        }

        public int length() {
            return end - start;
        }
    }

    private class BufferInputStream extends BAInputStream {
        private int position;
        private final int start;
        private final int end;

        private BufferInputStream(int start, int end) {
            this.start = start;
            this.end = end;
            this.position = start;
        }

        @Override
        public int read() {
            if (position == end) {
                return -1;
            }
            return buffer[position++] & 0xFF;
        }

        @Override
        public void resetPosition() {
            this.position = start;
        }

        @Override
        public void reset() {
            this.position = start;
        }
    }
}
