package ru.yandex.webmaster3.storage.util.clickhouse2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.storage.util.clickhouse2.query.QueryBuilder;
import ru.yandex.webmaster3.storage.util.clickhouse2.query.Where;

/**
 * Created by ifilippov5 on 17.04.17.
 */
public class MergeFromTempTableService extends AbstractClickhouseDao {
    private static final Logger log = LoggerFactory.getLogger(MergeFromTempTableService.class);

    public void mergeFrom(AbstractSupportMergeDataFromTempTableCHDao concreteClickhouseDao) throws ClickhouseException {
        final Map<ClickhouseHost, Set<String>> tablesWithPrefix = getTempTablesByClickhouseHost(concreteClickhouseDao);

        final Map<Pair<String, Integer>, List<ClickhouseHost>> hostsByTableWithShard = groupHostByTable(tablesWithPrefix);
        List<String> brokenTables = mergeTempTable(concreteClickhouseDao, hostsByTableWithShard);
        if (brokenTables.size() > 0) {
            throw new IllegalArgumentException("Some tables for merge are corrupted -- " + brokenTables);
        }
    }

    List<String> mergeTempTable(AbstractSupportMergeDataFromTempTableCHDao concreteClickhouseDao,
                               Map<Pair<String, Integer>, List<ClickhouseHost>> hostsByTableWithShards) {
        List<String> brokenTablesNames = new ArrayList<>();
        String dbName = concreteClickhouseDao.getDbName();

        hostsByTableWithShards.forEach((key, hosts) -> {
            final String tableName = key.getKey();
            log.info("Hosts : {}, Table : {}, Shard : {}", hosts, tableName, key.getValue());
//            logingCountRows(dbName, hosts, tableName);

            boolean hasUnbrokenTable = insertFromTempTable(concreteClickhouseDao, dbName, hosts, tableName);
            if (hasUnbrokenTable) {
                cleanProcessedTempTables(dbName, hosts, tableName);
            } else {
                brokenTablesNames.add(dbName + "." + tableName);
            }
        });
        return brokenTablesNames;
    }

    private void cleanProcessedTempTables(String dbName, List<ClickhouseHost> hosts, String tableName) {
        hosts.forEach(host -> {
            ClickhouseQueryContext.Builder chContext = ClickhouseQueryContext.useDefaults().setHost(host);
            String dropQuery = "DROP TABLE IF EXISTS " + dbName + "." + tableName;
            try {
                getClickhouseServer().execute(chContext, ClickhouseServer.QueryType.INSERT, dropQuery,
                    Optional.empty(), Optional.empty());
            } catch (ClickhouseException e) {
                log.error("Failed to delete table " + tableName + " from clickhouse host " + host.getHostURI(), e);
            }
        });
    }

    private boolean insertFromTempTable(AbstractSupportMergeDataFromTempTableCHDao concreteClickhouseDao, String dbName,
                                    List<ClickhouseHost> hosts, String tableName) {
        boolean unbrokenTable = false;
        for (ClickhouseHost host : hosts) {
            try {
                final boolean result = getClickhouseServer().executeWithFixedHost(host, () ->
                    queryOne(String.format("CHECK TABLE %s.%s", dbName, tableName), r -> r.getInt("result") > 0).orElse(false));
                if (!result) {
                    continue;
                }

                getClickhouseServer().executeWithFixedHost(host, () -> {
                        insertFrom(tableName, concreteClickhouseDao);
                        return null;
                    }
                );
                unbrokenTable = true;
                break;
            } catch (ClickhouseException e) {
                log.error("Failed to import from table " + tableName + " on clickhouse host " + host.getHostURI()
                        , e);
            }
        }
        return unbrokenTable;
    }

    private void logingCountRows(String dbName, List<ClickhouseHost> hosts, String tableName) {
        Map<ClickhouseHost, Long> cntRowsPerHost = new HashMap<>();
        for (ClickhouseHost host : hosts) {
            try {
                long res = getClickhouseServer().executeWithFixedHost(host, () ->
                        queryOne(String.format("SELECT count(*) as cnt FROM %s.%s", dbName, tableName), r -> r.getLongUnsafe("cnt")).orElse(0L)
                );
                cntRowsPerHost.put(host, res);
            } catch (Exception e) {
                log.info("Table on host : {} broken", host, e);
            }
        }
        log.info("Count rows per host: {}", cntRowsPerHost);
    }

    @NotNull
    Map<ClickhouseHost, Set<String>> getTempTablesByClickhouseHost(AbstractSupportMergeDataFromTempTableCHDao concreteClickhouseDao) {
        String tempTablePrefix = concreteClickhouseDao.getTempTablePrefix();
        String dbName = concreteClickhouseDao.getDbName();

        final long currentMinutesInterval = TempDataChunksStoreUtil.getCurrentMinutesInterval(concreteClickhouseDao.getMinutesIntervalSize());

        Where st = QueryBuilder.select(F.NAME)
            .from("system.tables")
            .where(QueryBuilder.eq(F.DATABASE, dbName))
            .and(QueryBuilder.startsWith(F.NAME, tempTablePrefix));


        return getClickhouseServer().getHosts().stream().filter(ClickhouseHost::isUp).collect(Collectors.toMap(x -> x, host -> {
            List<String> tablesByHost = Collections.emptyList();
            try {
                tablesByHost = getClickhouseServer().executeWithFixedHost(host, () -> queryAll(st, r -> r.getString(F.NAME))
                );
            } catch (ClickhouseException e) {
                log.error("Failed to show clickhouse tables", e);
            }

            // отсекаем таблицы, которые соответствуют текущему интервалу времени, так как в них в данный момент могут писаться данные
            return tablesByHost.stream()
                .filter(tableName -> TempDataChunksStoreUtil.extractTimeIntervalIdFromTempTableName(tableName, tempTablePrefix)
                    < currentMinutesInterval - 1)
                .collect(Collectors.toSet());
        }));
    }

    //               <tableName, shard> => hosts -- таблицы в одном шарде будут вместе.
    static Map<Pair<String, Integer>, List<ClickhouseHost>> groupHostByTable(Map<ClickhouseHost, Set<String>> tables) {
        Map<Pair<String, Integer>, List<ClickhouseHost>> results = new HashMap<>();
        tables.forEach((host, tablesNames) -> tablesNames.forEach(tableName -> {
                results.computeIfAbsent(Pair.of(tableName, host.getShard()), ign -> new ArrayList<>()).add(host);
            }
            )
        );
        return results;
    }

    private void insertFrom(String tableName, AbstractSupportMergeDataFromTempTableCHDao concreteClickhouseDao) throws ClickhouseException {
        concreteClickhouseDao.insertFromTempTable(tableName);
    }

    private static class F {
        static final String NAME = "name";
        static final String DATABASE = "database";
    }
}
