package ru.yandex.webmaster3.storage.util.clickhouse2;

import com.google.common.base.Charsets;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by ifilippov5 on 17.04.17.
 */
public class TempDataChunksStoreUtil {
    private static final Logger log = LoggerFactory.getLogger(TempDataChunksStoreUtil.class);

    /*** Выбирает живой хост среди всех. Какой именно хост будет выбран, зависит от текущего временного интервала.
     ***/
    public static ClickhouseHost selectHost(ClickhouseServer clickhouseServer, int minutesIntervalSize) throws ClickhouseException {
        List<ClickhouseHost> hosts = ClickhouseServer.alive().apply(clickhouseServer.getHosts());

        if (hosts.isEmpty()) {
            log.error("There are no alive hosts");
            return null;
        }

        hosts.sort(Comparator.comparingLong(host -> buildHash(host, minutesIntervalSize)));

        return hosts.get(0);
    }

    /*** Выбирает живой хост среди всех в пределах шарда. Какой именно хост будет выбран, зависит от текущего временного интервала.
     ***/
    public static ClickhouseHost selectHostForShard(int shard, ClickhouseServer clickhouseServer, int minutesIntervalSize) throws ClickhouseException {
        return Optional.ofNullable(selectKHostsForShard(shard, clickhouseServer, minutesIntervalSize, 1))
                .map(x -> x.get(0)).orElse(null);
    }

    /*** Выбирает живой хост среди всех в пределах шарда. Какой именно хост будет выбран, зависит от текущего временного интервала.
     ***/

    public static List<ClickhouseHost> selectKHostsForShard(int shard, ClickhouseServer clickhouseServer, int minutesIntervalSize, int k) throws ClickhouseException {
        List<ClickhouseHost> hosts = clickhouseServer.getHosts().stream()
                .filter(h -> h.getShard() == shard && h.isUp())
                .sorted(Comparator.comparingLong(host -> buildHash(host, minutesIntervalSize)))
                .limit(k)
                .collect(Collectors.toList());

        if (hosts.isEmpty()) {
            log.error("There are no alive hosts");
            return null;
        }

        return hosts;
    }
    /*** Возвращает идентификатор текущего временного интервала, который имеет размер minutesIntervalSize
     ***/
    public static long getCurrentMinutesInterval(int minutesIntervalSize) { // возможно стоит перенести в утилиты времени
        Date date = new Date();
        return date.getTime() / TimeUnit.MINUTES.toMillis(minutesIntervalSize);
    }

    /*** tableName = prefix + идентификатор временного интервала
     ***/
    public static long extractTimeIntervalIdFromTempTableName(String tableName, String prefix) {
        if (tableName.length() < prefix.length()) {
            throw new IllegalArgumentException("Name of " + tableName + " not contains prefix = " + prefix);
        }
        String tablePrefix = tableName.substring(0, prefix.length());
        if (!tablePrefix.equals(prefix)) {
            throw new IllegalArgumentException("Name of " + tableName + " not contains prefix = " + prefix);
        }

        String tableSuffix = tableName.substring(prefix.length());
        return Long.valueOf(tableSuffix);
    }

    private static long buildHash(ClickhouseHost host, int minutesIntervalSize) {
        HashFunction hf = Hashing.md5();
        Hasher hc = hf.newHasher()
                .putString(host.getHostURI().toString(), Charsets.UTF_8) // считаем, что в различных датацентрах uri хостов различаются
                .putLong(getCurrentMinutesInterval(minutesIntervalSize));

        return hc.hash().asLong();
    }
}
