package ru.yandex.webmaster3.storage.util.sql;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.core.RowMapper;
import ru.yandex.webmaster3.storage.WebmasterSQLException;
import ru.yandex.wmtools.common.util.ParameterizedMapRowMapper;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;

/**
 * @author avhaliullin
 */
public abstract class DelegatingReadJdbcTemplate implements IReadJdbcTemplate {
    protected final Logger log = LoggerFactory.getLogger(getClass());

    protected abstract IReadJdbcTemplate getTemplate();

    protected abstract String prepareQuery(String s);

    protected abstract void onQueryStart(StartQueryInfo info);

    protected abstract void onQueryFinish(FinishQueryInfo info);

    protected void beforeFinish() {
    }

    @Override
    public String getDbIdentity() {
        return getTemplate().getDbIdentity();
    }

    @Override
    public int queryForInt(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Integer>(getTemplate(), sqlString) {
            @Override
            public Integer doQuery() throws WebmasterSQLException {
                return template.queryForInt(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public long queryForLong(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Long>(getTemplate(), sqlString) {
            @Override
            public Long doQuery() throws WebmasterSQLException {
                return template.queryForLong(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public <T> T queryForObject(final String sqlString, final Class<T> aClass, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<T>(getTemplate(), sqlString) {
            @Override
            public T doQuery() throws WebmasterSQLException {
                return template.queryForObject(prepareQuery(sqlString), aClass, requestParams);
            }
        }.query();
    }

    @Override
    public <T> T queryForObject(final String sqlString, final RowMapper<T> rowMapper, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<T>(getTemplate(), sqlString) {
            @Override
            public T doQuery() throws WebmasterSQLException {
                return template.queryForObject(prepareQuery(sqlString), rowMapper, requestParams);
            }
        }.query();
    }

    @Override
    public <T> List<T> query(final String sqlString, final RowMapper<T> rowMapper, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<List<T>>(getTemplate(), sqlString) {
            @Override
            public List<T> doQuery() throws WebmasterSQLException {
                return template.query(prepareQuery(sqlString), rowMapper, requestParams);
            }
        }.query();
    }

    @Override
    public Map<String, Object> queryForMap(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Map<String, Object>>(getTemplate(), sqlString) {
            @Override
            public Map<String, Object> doQuery() throws WebmasterSQLException {
                return template.queryForMap(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public List<Map<String, Object>> queryForList(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<List<Map<String, Object>>>(getTemplate(), sqlString) {
            @Override
            public List<Map<String, Object>> doQuery() throws WebmasterSQLException {
                return template.queryForList(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public Date safeQueryForTimestamp(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Date>(getTemplate(), sqlString) {
            @Override
            public Date doQuery() throws WebmasterSQLException {
                return template.safeQueryForTimestamp(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public Integer safeQueryForInt(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Integer>(getTemplate(), sqlString) {
            @Override
            public Integer doQuery() throws WebmasterSQLException {
                return template.safeQueryForInt(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public Long safeQueryForLong(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<Long>(getTemplate(), sqlString) {
            @Override
            public Long doQuery() throws WebmasterSQLException {
                return template.safeQueryForLong(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public String safeQueryForString(final String sqlString, final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<String>(getTemplate(), sqlString) {
            @Override
            public String doQuery() throws WebmasterSQLException {
                return template.safeQueryForString(prepareQuery(sqlString), requestParams);
            }
        }.query();
    }

    @Override
    public <E> E safeQueryForObject(final String sqlString, final RowMapper<E> rowMapper,
                                    final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<E>(getTemplate(), sqlString) {
            @Override
            public E doQuery() throws WebmasterSQLException {
                return template.safeQueryForObject(prepareQuery(sqlString), rowMapper, requestParams);
            }
        }.query();
    }

    @Override
    public <K, V> NavigableMap<K, V> queryForNavigableMap(final String sqlString,
                                                          final ParameterizedMapRowMapper<K, V> rowMapper,
                                                          final Object... requestParams) throws WebmasterSQLException {
        return new ReadQueryProcessor<NavigableMap<K, V>>(getTemplate(), sqlString) {
            @Override
            public NavigableMap<K, V> doQuery() throws WebmasterSQLException {
                return template.queryForNavigableMap(prepareQuery(sqlString), rowMapper, requestParams);
            }
        }.query();
    }

    @Override
    public void query(final String sqlString, final RowCallbackHandler rch, final Object... requestParams) throws WebmasterSQLException {
        new ReadQueryProcessor<Void>(getTemplate(), sqlString) {
            @Override
            public Void doQuery() throws WebmasterSQLException {
                getTemplate().query(prepareQuery(sqlString), rch, requestParams);
                return null;
            }
        }.query();
    }

    public class StartQueryInfo {
        public final String query;
        public final boolean write;
        public final IReadJdbcTemplate template;

        public StartQueryInfo(String query, boolean write, IReadJdbcTemplate template) {
            this.query = query;
            this.write = write;
            this.template = template;
        }
    }

    public class FinishQueryInfo {
        public final StartQueryInfo queryInfo;
        public final long executionTimeMs;
        public final Throwable exception;

        public FinishQueryInfo(StartQueryInfo queryInfo, long executionTimeMs, Throwable exception) {
            this.queryInfo = queryInfo;
            this.executionTimeMs = executionTimeMs;
            this.exception = exception;
        }
    }

    protected abstract class ReadQueryProcessor<T> {
        protected final IReadJdbcTemplate template;
        protected final String query;

        protected ReadQueryProcessor(IReadJdbcTemplate template, String query) {
            this.template = template;
            this.query = query;
        }

        public T query() throws WebmasterSQLException {
            long startTime = System.currentTimeMillis();
            StartQueryInfo startInfo = new StartQueryInfo(query, false, template);
            onQueryStart(startInfo);
            try {
                T result = doQuery();
                beforeFinish();
                onQueryFinish(new FinishQueryInfo(startInfo, System.currentTimeMillis() - startTime, null));
                return result;
            } catch (Throwable e) {
                onQueryFinish(new FinishQueryInfo(startInfo, System.currentTimeMillis() - startTime, e));
                throw e;
            }
        }

        public abstract T doQuery() throws WebmasterSQLException;
    }
}
