package ru.yandex.webmaster3.storage.util.ydb;

import java.time.Duration;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.core.auth.TokenAuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

/**
 * ishalaru
 * 17.06.2020
 **/
@Configuration
public class Config {
    private static final int MIN_SESSION_POOL_SIZE = 150;
    private static final int MAX_SESSION_POOL_SIZE = 1000;

    @Bean(name = "ydbRpcTransport")
    public GrpcTransport ydbRpcTransport(@Qualifier("ydbCallTaskExecutor") ThreadPoolTaskExecutor threadPoolTaskExecutor,
                                        @Value("${webmaster3.storage.ydb.endpoint}") String endPoint,
                                        @Value("${webmaster3.storage.ydb.dbname}") String dbName,
                                        @Value("${webmaster3.storage.ydb.auth.token}") String authToken,
                                        @Value("${webmaster3.storage.ydb.session.read.timeout:600000}") long readTimeout) {
        return GrpcTransport.forEndpoint(endPoint, dbName)
                .withAuthProvider(new TokenAuthProvider(authToken))
                .withReadTimeout(readTimeout, TimeUnit.MILLISECONDS)
                .withCallExecutor(threadPoolTaskExecutor)
                .build();
    }

    @Bean(name = "tableClient", destroyMethod = "close")
    public TableClient tableClient(@Qualifier("ydbRpcTransport") RpcTransport ydbRpcTransport) {
        return TableClient.newClient(GrpcTableRpc.ownTransport(ydbRpcTransport)).sessionPoolSize(MIN_SESSION_POOL_SIZE, MAX_SESSION_POOL_SIZE).build();
    }


    @Bean("ydbCallTaskExecutor")
    public ThreadPoolTaskExecutor ydbCallTaskExecutor() {
        ThreadPoolTaskExecutor executor = new ThreadPoolTaskExecutor();
        executor.setQueueCapacity(10000);
        executor.setMaxPoolSize(30);
        executor.setCorePoolSize(3);
        executor.setDaemon(true);
        executor.setThreadNamePrefix("ydb-call-executor-");
        executor.setAwaitTerminationSeconds(1);
        executor.setWaitForTasksToCompleteOnShutdown(true);
        return executor;
    }

    @Bean("ydbSessionRetryContext")
    public SessionRetryContext ydbSessionRetryContext(@Qualifier("tableClient") TableClient tableClient) {
        return SessionRetryContext.create(tableClient)
                .executor(Executors.newWorkStealingPool())
                .maxRetries(5)
                .sessionSupplyTimeout(Duration.ofSeconds(3))
                .build();
    }

    @Bean(name = "ydbAsyncMutex")
    @Scope(value = "singleton")
    public Semaphore mutex() {
        return new Semaphore(2 * MAX_SESSION_POOL_SIZE, false);
    }

    @Bean(name = "waitAcquireDuration")
    @Scope(value = "singleton")
    public Duration waitAcquireDuration() {
        return Duration.ofSeconds(5);
    }

    @Bean(name = "maxWaitingTimeout")
    public Duration maxWaitingTimeout() {
        return Duration.ofSeconds(30);
    }
}
