package ru.yandex.webmaster3.storage.util.ydb.query;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;
import org.apache.commons.lang3.tuple.Pair;

/**
 * ishalaru
 * 11.06.2020
 **/
public abstract class Clause extends Utils.Appendeable implements ParametersDeclaration {

    public abstract void putParameter(Map<String, Value> map);

    public abstract int initIndex(int index);

    private abstract static class AbstractClause extends Clause {
        final String name;
        private int parameterIndex;

        private AbstractClause(String name) {
            this.name = name;
        }

        protected String name() {
            return name;
        }

        protected int getParameterIndex() {
            return parameterIndex;
        }

        @Override
        public int initIndex(int index) {
            this.parameterIndex = index;
            index++;
            return index;
        }
    }

    public static class SimpleClause extends AbstractClause {
        private final String op;
        private final Pair<Type, Value> value;

        public SimpleClause(String name, String op, Object value) {
            super(name);
            this.op = op;
            this.value = Utils.defineType(value);
        }

        @Override
        public void putParameter(Map<String, Value> map) {
            map.put(Utils.generateParameterName(getParameterIndex()), value.getValue());
        }


        @Override
        public void appendTo(StringBuilder sb) {
            Utils.appendName(name(), sb);
            sb.append(" " + op + " ");
            Utils.appendValue(getParameterIndex(), sb);
        }

        @Override
        public StringBuilder appendDeclaration(StringBuilder sb, Set<Integer> declaredParameters) {
            if (!declaredParameters.contains(getParameterIndex())) {
                declaredParameters.add(getParameterIndex());
                return sb.append("DECLARE ")
                        .append(Utils.generateParameterName(getParameterIndex()))
                        .append(" as ")
                        .append(value.getLeft().toString())
                        .append(";\n");
            }
            return sb;
        }
    }

    @lombok.Value
    public static class BinaryClause extends Clause {
        Clause clauseA;
        Clause clauseB;
        String operation;

        @Override
        public void putParameter(Map<String, Value> map) {
            clauseA.putParameter(map);
            clauseB.putParameter(map);
        }

        @Override
        public int initIndex(int index) {
            int answer = clauseA.initIndex(index);
            answer = clauseB.initIndex(answer);
            return answer;
        }

        @Override
        public StringBuilder appendDeclaration(StringBuilder sb, Set<Integer> declaredParameters) {
            StringBuilder result = clauseA.appendDeclaration(sb, declaredParameters);
            result = clauseB.appendDeclaration(result, declaredParameters);
            return result;
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append("(");
            clauseA.appendTo(sb);
            sb.append(" " + operation + " ");
            clauseB.appendTo(sb);
            sb.append(" ) ");
        }
    }

    @lombok.Value
    public static class MultiClause extends Clause {
        Collection<Clause> clauses;
        String operation;

        @Override
        public void putParameter(Map<String, Value> map) {
            for (Clause clause : clauses) {
                clause.putParameter(map);
            }
        }

        @Override
        public int initIndex(int index) {
            for (Clause clause : clauses) {
                index = clause.initIndex(index);
            }
            return index;
        }

        @Override
        public StringBuilder appendDeclaration(StringBuilder sb, Set<Integer> declaredParameters) {
            for (Clause clause : clauses) {
                sb = clause.appendDeclaration(sb, declaredParameters);
            }
            return sb;
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append("(");
            boolean first = true;
            for (Clause clause : clauses) {
                if (!first) {
                    sb.append(" " + operation + " ");
                }
                clause.appendTo(sb);
                first = false;
            }
            sb.append(" ) ");
        }
    }

    public static class InClause extends AbstractClause {
        private final Pair<Type, Value> value;

        public InClause(String name, Iterable<?> value) {
            super(name);
            this.value = Utils.defineType(value);
        }

        @Override
        public void putParameter(Map<String, Value> map) {
            map.put(Utils.generateParameterName(getParameterIndex()), value.getValue());
        }

        @Override
        public StringBuilder appendDeclaration(StringBuilder sb, Set<Integer> declaredParameters) {
            if (!declaredParameters.contains(getParameterIndex())) {
                declaredParameters.add(getParameterIndex());
                return sb.append("DECLARE ")
                        .append(Utils.generateParameterName(getParameterIndex()))
                        .append(" as ")
                        .append(value.getLeft().toString())
                        .append(";\n");
            }
            return sb;
        }

        @Override
        public void appendTo(StringBuilder sb) {
            Utils.appendName(name(), sb);
            sb.append(" in ");
            Utils.appendValue(getParameterIndex(), sb);
        }
    }

    protected abstract static class NoParameterClause extends AbstractClause {

        private NoParameterClause(String name) {
            super(name);
        }

        @Override
        public void putParameter(Map<String, Value> map) {
            //do nothing
        }

        @Override
        public int initIndex(int index) {
            return index;
        }

        @Override
        public StringBuilder appendDeclaration(StringBuilder sb, Set<Integer> declaredParameters) {
            return sb;
        }
    }

    public static class IsNullClause extends NoParameterClause {

        public IsNullClause(String name) {
            super(name);
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append(name).append(" IS NULL\n");
        }
    }

    public static class IsNotNullClause extends NoParameterClause {

        public IsNotNullClause(String name) {
            super(name);
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append(name).append(" IS NOT NULL\n");
        }
    }

    public static class TrueClause extends NoParameterClause {

        public TrueClause() {
            super("TRUE");
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append("True");
        }
    }

    public static class FalseClause extends NoParameterClause {

        public FalseClause() {
            super("FALSE");
        }

        @Override
        public void appendTo(StringBuilder sb) {
            sb.append("False");
        }
    }
}
