package ru.yandex.webmaster3.storage.util.ydb.querybuilder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.webmaster3.storage.util.ydb.SelectQuery;
import ru.yandex.webmaster3.storage.util.ydb.query.Clause;
import ru.yandex.webmaster3.storage.util.ydb.query.Statement;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.FieldMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.RowMapper;

/**
 * ishalaru
 * 05.06.2020
 **/
@Slf4j
public class Select<T> extends Statement {
    private Set<String> columnNames = new LinkedHashSet<>();
    private int index;
    private Where<T> where;
    private OrderBy<T> order;
    private Limit<T> limit;
    private GroupBy<T> groupBy;
    private Having<T> having;
    private Cont<T> cont;
    private SecondaryIndex<T> secondaryIndex;
    private CountAll countAll = null;
    private RowMapper<T> rowMapper;
    private SelectQuery selectQuery;
    private boolean isDistinct;

    public Select(String tablePrefix,
                  String table, RowMapper<T> rowMapper,
                  SelectQuery selectQuery) {
        super(tablePrefix, table, OperationType.SELECT);
        this.index = 1;
        this.rowMapper = rowMapper;
        if (rowMapper != null && rowMapper.getDependencyColumns() != null) {
            rowMapper.getDependencyColumns().forEach(columnName -> this.columnNames.add(columnName));
        }
        this.where = new Where(this);
        this.order = new OrderBy(this);
        this.limit = new Limit(this);
        this.groupBy = new GroupBy(this);
        this.having = new Having(this);
        this.secondaryIndex = new SecondaryIndex(this);
        this.cont = new Cont(this);
        this.selectQuery = selectQuery;
        this.isDistinct = false;
    }


    public Where<T> where(Clause condition) {
        return where.and(condition);
    }

    public GroupBy<T> groupBy(Field... fields) {
        for (Field field : fields) {
            groupBy.add(field);
        }
        return groupBy;
    }

    public Having<T> having(Clause condition) {
        return having.and(condition);
    }

    public Select<T> order(Ordering ordering) {
        columnNames.add(ordering.getName());
        return order.add(ordering);
    }

    public Select<T> limit(int limitVal) {
        limit.setLimit(limitVal);
        return this;
    }

    public Select<T> limit(int skipVal, int limitVal) {
        limit.setLimit(skipVal, limitVal);
        return this;
    }

    public Select<T> countAll() {
        this.countAll = new CountAll(this);
        return this;
    }

    public Select<T> distinct() {
        this.isDistinct = true;
        return this;
    }

    /**
     * В строку запроса добавляется как "AND (condition)"
     */
    public Cont<T> cont(Clause condition) {
        cont = new Cont(this);
        return cont.and(condition);
    }

    /**
     * Для того чтобы использовался вторичный индекс его использование нужно указать явно.
     * Использование вторичного индекса возможно только при использовании v1 синтаксиса.
     *
     * @param secondaryIndexName название вторичного индекса
     * @return возвращает объект Select для дальнейшего конструирования запроса
     */
    public Select<T> secondaryIndex(String secondaryIndexName) {
        secondaryIndex.setSecondaryIndex(secondaryIndexName);
        useFirstVersion = true;
        return this;
    }

    private String getQueryHeader() {
        StringBuilder sb = new StringBuilder();
        //if (useFirstVersion) {
        sb.append(SYNTAX_FIRST_VERSION).append('\n');
        //}
        sb.append(TABLE_PREFIX).append("'").append(tablePrefix).append("';\n");

        appendDeclarations(sb);

        return sb.toString();
    }

    void appendDeclarations(StringBuilder sb) {
        if (!where.list.isEmpty()) {
            where.appendDeclaration(sb);
        }

        if (!cont.where.list.isEmpty()) {
            cont.where.appendDeclaration(sb);
        }

        if (!having.where.list.isEmpty()) {
            having.where.appendDeclaration(sb);
        }

        if (limit.hasLimits()) {
            limit.appendDeclaration(sb);
        }
    }

    String getQueryBody() {
        StringBuilder sb = new StringBuilder();
        sb.append("SELECT ");
        if (isDistinct) {
            sb.append("DISTINCT ");
        }
        sb.append('\n');

        if (columnNames.isEmpty()) {
            sb.append("*");
        } else {
            for (String column : columnNames) {
                // TODO нужен нормальный фреймворк или вообще отказаться от query-builder-ов
                if (column.equals("count(*)")) {
                    sb.append("count(*),\n");
                } else {
                    sb.append("`").append(column).append("`,\n");
                }
            }
            sb.setLength(sb.length() - 2);
        }
        sb.append("\n");
        sb.append("FROM ").append(table);
        if (secondaryIndex.secondaryIndex != null) {
            secondaryIndex.append(sb);
        }
        sb.append("\n");
        if (!where.list.isEmpty()) {
            sb.append("WHERE \n");
            where.append(sb);
        }
        if (!cont.where.list.isEmpty()) {
            sb.append(" AND (\n");
            cont.where.append(sb);
            sb.append(" )\n");
        }
        if (!groupBy.columnNames.isEmpty()) {
            sb.append("GROUP BY ");
            groupBy.append(sb);
            sb.append("\n");
        }
        if (!having.where.list.isEmpty()) {
            sb.append("HAVING \n");
            having.where.append(sb);
        }
        if (!order.list.isEmpty()) {
            sb.append("ORDER BY ");
            order.append(sb);
        }
        if (limit.hasLimits()) {
            limit.append(sb);
        }

        return sb.toString();
    }

    @Override
    public String toQueryString() {
        if (countAll != null) {
            return getQueryHeader() + "\n" +
                    "SELECT count(*) FROM (" +
                    getQueryBody() +
                    ");";
        } else {
            return getQueryHeader() +
                    getQueryBody() +
                    ";";
        }
    }

    @Override
    public Map<String, Value> getParameters() {
        Map<String, Value> result = new HashMap<>();
        where.list.forEach(e -> e.putParameter(result));
        cont.where.list.forEach(e -> e.putParameter(result));
        having.where.list.forEach(e -> e.putParameter(result));
        if (limit.hasLimits()) {
            result.put("$skip", PrimitiveValue.int32(limit.skip));
            result.put("$limit", PrimitiveValue.int32(limit.limit));
        }
        return result;
    }

    // вытаскивает до 1000 записей
    public List<T> queryForList() {
        return selectQuery.queryForList(this, rowMapper);
    }

    public List<T> queryForList(Function<List<T>, Select<T>> continuationSupplier) {
        return selectQuery.queryForList(this, rowMapper, continuationSupplier);
    }

    public <KEY> List<T> queryForList(Pair<Field<KEY>, Function<T, KEY>> key) {
        return queryForList(FieldMapper.create(key.getLeft(), key.getRight()));
    }

    public <KEY1, KEY2> List<T> queryForList(Pair<Field<KEY1>, Function<T, KEY1>> key1, Pair<Field<KEY2>, Function<T, KEY2>> key2) {
        return queryForList(FieldMapper.create(key1.getLeft(), key1.getRight()), FieldMapper.create(key2.getLeft(), key2.getRight()));
    }

    public <KEY1, KEY2, KEY3> List<T> queryForList(Pair<Field<KEY1>, Function<T, KEY1>> key1,
                                                   Pair<Field<KEY2>, Function<T, KEY2>> key2,
                                                   Pair<Field<KEY3>, Function<T, KEY3>> key3) {
        return queryForList(FieldMapper.create(key1.getLeft(), key1.getRight()),
                FieldMapper.create(key2.getLeft(), key2.getRight()),
                FieldMapper.create(key3.getLeft(), key3.getRight()));
    }

    /**
     * Добавляет кучу сортировок для преодоления лимита в 1000 записей от ydb
     * select ...
     * from ...
     * where (f1 > v1) or (f1 = v1 and f2 > v2) or (f1 = v1 and f2 = v2 and f3 > v3) ...
     * order by f1, f2, f3, f4 ...
     *
     * @param mappers
     * @return
     */
    public List queryForList(FieldMapper... mappers) {
        Preconditions.checkState(order.list.isEmpty(), "queryForList - unexpect ORDER ");

        for (int i = 0; i < mappers.length; i++) {
            order(mappers[i].getField().asc());
        }
        // так как cont добавляет только AND
        where(QueryBuilder.addTrue());
        return selectQuery.queryForList(this, rowMapper, list -> {
            var last = Iterables.getLast(list);
            List<Clause> ors = new ArrayList<>();
            for (int i = 0; i < mappers.length; i++) {
                List<Clause> ands = new ArrayList<>();
                for (int j = 0; j < i; j++) {
                    ands.add(mappers[j].getField().eq(mappers[j].getMapper().apply(last)));
                }
                ands.add(mappers[i].getField().gt(mappers[i].getMapper().apply(last)));
                ors.add(QueryBuilder.and(ands));
            }
            return cont(QueryBuilder.or(ors)).getStatement();
        });
    }

    public T queryOne() {
        return selectQuery.queryOne(this, rowMapper);
    }

    public static class Where<T> extends SelectBaseOperation<T> {
        List<Clause> list;

        public Where(Select parent) {
            super(parent);
            list = new ArrayList<>(6);
        }

        public Where<T> and(Clause condition) {
            parent.index = condition.initIndex(parent.index);
            list.add(condition);
            return this;
        }

        public StringBuilder appendDeclaration(StringBuilder sb) {
            Set<Integer> declaredParameter = new HashSet<>();
            for (Clause clause : list) {
                clause.appendDeclaration(sb, declaredParameter);
            }
            return sb;
        }

        StringBuilder append(StringBuilder sb) {
            for (int i = 0; i < list.size(); i++) {
                Clause clause = list.get(i);
                if (i == list.size() - 1) {
                    clause.appendTo(sb);
                    sb.append("\n");
                } else {
                    clause.appendTo(sb);
                    sb.append(" and \n");
                }
            }
            return sb;
        }
    }

    public static class OrderBy<T> extends SelectBaseOperation<T> {
        List<Ordering> list;


        public OrderBy(Select<T> parent) {
            super(parent);
            list = new ArrayList<>();
        }

        public Select<T> add(Ordering ordering) {
            list.add(ordering);
            return parent;
        }

        StringBuilder append(StringBuilder sb) {
            for (Ordering ordering : list) {
                ordering.append(sb).append(",");
            }
            sb.setLength(sb.length() - 1);
            sb.append("\n");
            return sb;
        }
    }

    public static class Limit<T> extends SelectBaseOperation<T> {
        int skip;
        int limit;
        boolean limitSet;

        public Limit(Select parent) {
            super(parent);

            this.skip = 0;
            this.limit = 0;
            this.limitSet = false;
        }

        public void setLimit(int limit) {
            this.skip = 0;
            this.limit = limit;
            this.limitSet = true;
        }

        public void setLimit(int skip, int limit) {
            this.skip = skip;
            this.limit = limit;
            this.limitSet = true;
        }

        boolean hasLimits() {
            return limitSet;
        }

        StringBuilder append(StringBuilder sb) {
            return sb.append("LIMIT $skip, $limit\n");
        }

        public StringBuilder appendDeclaration(StringBuilder sb) {
            sb.append("DECLARE $skip as Int32;\n");
            sb.append("DECLARE $limit as Int32;\n");

            return sb;
        }
    }

    public static class GroupBy<T> extends SelectBaseOperation<T> {
        List<Field> columnNames;

        public GroupBy(Select parent) {
            super(parent);
            this.columnNames = new ArrayList<>(2);
        }

        public GroupBy<T> add(Field columnName) {
            this.columnNames.add(columnName);
            return this;
        }

        StringBuilder append(StringBuilder sb) {
            for (Field a : columnNames) {
                sb.append("`").append(a.getName()).append("`").append(",");
            }
            sb.setLength(sb.length() - 1);
            return sb;
        }
    }

    public static class Having<T> extends SelectBaseOperation<T> {
        Where<T> where;

        public Having(Select<T> parent) {
            super(parent);
            where = new Where<T>(parent);
        }

        public Having<T> and(Clause condition) {
            where.and(condition);
            return this;
        }

        public StringBuilder appendDeclaration(StringBuilder sb) {
            return where.appendDeclaration(sb);
        }

        StringBuilder append(StringBuilder sb) {
            return where.append(sb);
        }

    }

    // Типа Where, но для пагинации, когда упираемся в лимит числа строк в ответе
    public static class Cont<T> extends SelectBaseOperation<T> {
        Where<T> where;

        public Cont(Select<T> parent) {
            super(parent);
            where = new Where(parent);
        }

        public Cont<T> and(Clause condition) {
            where.and(condition);
            return this;
        }

        public StringBuilder appendDeclaration(StringBuilder sb) {
            return where.appendDeclaration(sb);
        }

        StringBuilder append(StringBuilder sb) {
            return where.append(sb);
        }
    }

    @Setter
    public static class SecondaryIndex<T> extends SelectBaseOperation<T> {

        private String secondaryIndex;

        public SecondaryIndex(Select<T> parent) {
            super(parent);
        }

        StringBuilder append(StringBuilder sb) {
            return sb.append(" view ").append(secondaryIndex);
        }
    }

    public static class CountAll extends SelectBaseOperation {
        public CountAll(Select parent) {
            super(parent);
        }
    }
}
