package ru.yandex.webmaster3.storage.util.yt;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.*;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.util.functional.ThrowingConsumer;
import ru.yandex.webmaster3.storage.util.yt.operation.YtAbortOperationCommand;
import ru.yandex.webmaster3.storage.util.yt.operation.YtCompleteOperationCommand;

/**
 * @author aherman
 */
public class YtCypressServiceImpl implements YtCypressService {
    private static final Logger log = LoggerFactory.getLogger(YtCypressServiceImpl.class);

    private static final String ATTRIBUTE_NODE_TYPE = "type";
    private static final String ATTRIBUTE_NODE_CREATION_TIME = "creation_time";
    private static final String ATTRIBUTE_NODE_MODIFICATION_TIME = "modification_time";
    private static final String ATTRIBUTE_NODE_OWNER = "owner";
    private static final String ATTRIBUTE_NODE_UNCOMPRESSED_DATA_SIZE = "uncompressed_data_size";
    private static final String ATTRIBUTE_NODE_COMPRESSED_DATA_SIZE = "compressed_data_size";
    private static final String ATTRIBUTE_NODE_SORTED = "sorted";
    private static final String ATTRIBUTE_NODE_SORTED_BY = "sorted_by";
    private static final String ATTRIBUTE_NODE_ROW_COUNT = "row_count";
    private static final String ATTRIBUTE_NODE_SCHEMA = "schema";

    private static final String ATTRIBUTE_OPERATION_STATE = "state";
    private static final String ATTRIBUTE_OPERATION_START_TIME = "start_time";
    private static final String ATTRIBUTE_OPERATION_PROGRESS = "progress";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS = "jobs";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS_TOTAL = "total";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS_PENDING = "pending";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS_RUNNING = "running";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS_COMPLETED = "completed";
    private static final String ATTRIBUTE_OPERATION_PROGRESS_JOBS_COMPLETED_NON_INTERRUPTED = "non-interrupted";

    final YtService ytService;
    final YtTransaction transaction;

    public YtCypressServiceImpl(YtService ytService, YtTransaction transaction) {
        this.ytService = ytService;
        this.transaction = transaction;
    }

    @Override
    public String getTransactionId() {
        return transaction == null ? null : transaction.getId();
    }

    @Override
    public YtNode getNode(YtPath path) throws YtException {
        YtGetAttributesCommand command = new YtGetAttributesCommand(path);
        YtResult<JsonNode> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        if (result.getStatus() == YtStatus.YT_400_FINISHED_WITH_ERROR) {
            log.error("Unable to get node: {}", result.getStatus());
            return null;
        }
        ytService.throwIfError(result, command);
        return createNode(path, result.getResult());
    }

    private YtNode createNode(YtPath path, JsonNode jsonNode) {
        String id = jsonNode.get("id").textValue();
        YtNode.NodeType nodeType = YtNode.NodeType.getType(jsonNode.get(ATTRIBUTE_NODE_TYPE).textValue());
        DateTime creationTime = new DateTime(jsonNode.get(ATTRIBUTE_NODE_CREATION_TIME).asText());
        DateTime updateTime = new DateTime(jsonNode.get(ATTRIBUTE_NODE_MODIFICATION_TIME).asText());
        String owner = jsonNode.get(ATTRIBUTE_NODE_OWNER).asText();
        if (nodeType != YtNode.NodeType.TABLE) {
            return new YtNode(path, nodeType, id, owner, creationTime, updateTime, jsonNode);
        }

        long uncompressedDataSize = jsonNode.get(ATTRIBUTE_NODE_UNCOMPRESSED_DATA_SIZE).asLong();
        long compressedDataSize = jsonNode.get(ATTRIBUTE_NODE_COMPRESSED_DATA_SIZE).asLong();

        boolean sorted = jsonNode.get(ATTRIBUTE_NODE_SORTED).asBoolean();
        List<String> sortedBy = Collections.emptyList();
        if (sorted) {
            JsonNode sortedByNode = jsonNode.get(ATTRIBUTE_NODE_SORTED_BY);
            sortedBy = YtJsonUtils.extractStringArray(sortedByNode);
        }

        JsonNode rowCountNode = jsonNode.get(ATTRIBUTE_NODE_ROW_COUNT);
        long rowCount = rowCountNode == null? 0 : rowCountNode.longValue();

        YtSchema schema = null;
        JsonNode schemaNode = jsonNode.get(ATTRIBUTE_NODE_SCHEMA);
        if (schemaNode != null && !schemaNode.isNull()) {
            schema = YtSchema.fromJsonNode(schemaNode);
        }

        return new YtTable(path, nodeType, id, owner, creationTime, updateTime, sorted, sortedBy, rowCount,
                uncompressedDataSize, compressedDataSize, schema, jsonNode);
    }

    @Override
    public boolean exists(YtPath path) throws YtException {
        YtExistsCommand command = new YtExistsCommand(path);
        YtResult<Boolean> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public YtNode create(YtPath path, YtNode.NodeType nodeType, boolean recursive) throws YtException {
        return create(path, nodeType, recursive, null);
    }

    @Override
    public YtNode create(YtPath path, YtNode.NodeType nodeType, boolean recursive, YtNodeAttributes attributes) throws YtException {
        return create(path, nodeType, recursive, attributes, false);
    }

    @Override
    public YtNode create(YtPath path, YtNode.NodeType nodeType, boolean recursive, YtNodeAttributes attributes,
                         boolean ignoreExisting) throws YtException {
        YtCreateCommand command = new YtCreateCommand(path, nodeType, attributes, ignoreExisting);
        command.setRecursive(recursive);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return getNode(path);
    }

    @Override
    public YtNode createReplicatedTable(YtPath replicatedTablePath, YtSchema schema,
                                        boolean ignoreExisting, boolean mountTable) throws YtException {
        YtNodeAttributes attributes = new YtNodeAttributes().setSchema(schema);
        attributes.addAttribute("dynamic", true);
        YtNode node = create(replicatedTablePath, YtNode.NodeType.REPLICATED_TABLE, false, attributes, ignoreExisting);

        if (mountTable) {
            mount(replicatedTablePath);
        }

        return node;
    }

    @Override
    public YtReplicaId setupReplication(YtPath replicatedTablePath, YtPath replicaPath, YtSchema schema,
                                 boolean ignoreExisting, boolean mountReplica) throws YtException {
        // Создание записи про реплику на мета-кластере
        YtNodeAttributes attributes = new YtNodeAttributes().setSchema(schema);
        YtReplicaId replicaId = createReplica(replicatedTablePath, replicaPath, attributes);

        // Создание непосредственно таблицы-реплики
        attributes.addAttribute("upstream_replica_id", replicaId.getId());
        attributes.addAttribute("dynamic", true);
        create(replicaPath, YtNode.NodeType.TABLE, false, attributes, ignoreExisting);

        if (mountReplica) {
            mount(replicaPath);
        }

        return replicaId;
    }

    @Override
    public void alterTableReplica(YtReplicaId replicaId, @Nullable Boolean isEnabled,
                           @Nullable YtAlterTableReplicaCommand.ReplicationMode replicationMode) throws YtException {
        YtAlterTableReplicaCommand command = new YtAlterTableReplicaCommand(replicaId, isEnabled, replicationMode);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(replicaId.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void convertTableToDynamic(YtPath path) throws YtException {
        YtConvertTableToDynamicCommand command = new YtConvertTableToDynamicCommand(path);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public YtReplicaId createReplica(YtPath tablePath, YtPath replicaPath, YtNodeAttributes attributes) throws YtException {
        YtCreateReplicaCommand command = new YtCreateReplicaCommand(tablePath, replicaPath, attributes);
        YtResult<String> result = ytService.execute(transaction, ytService.getCluster(tablePath.getCluster()), command);
        ytService.throwIfError(result, command);
        return new YtReplicaId(replicaPath.getCluster(), result.getResult());
    }

    public void mount(YtPath path) throws YtException {
        YtMountCommand command = new YtMountCommand(path);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);

        String tabletState = getTabletState(path);
        while (!tabletState.equals("mounted")) {
            try {
                Thread.sleep(TimeUnit.SECONDS.toMillis(1));
            } catch (InterruptedException e) {
                throw new YtException("Interrupted while waiting for operation end", e);
            }

            tabletState = getTabletState(path);
        }
    }

    private String getTabletState(YtPath path) {
        YtNode node = getNode(path);
        return node.getNodeMeta().get("tablet_state").asText();
    }

    @Override
    public List<YtPath> list(YtPath path) throws YtException {
        YtListCommand command = new YtListCommand(path);
        YtResult<List<YtPath>> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public void remove(YtPath path) throws YtException {
        remove(path, false, false);
    }

    @Override
    public void remove(YtPath path, boolean recursive) throws YtException {
        remove(path, recursive, false);
    }

    @Override
    public void remove(YtPath path, boolean recursive, boolean force) throws YtException {
        YtRemoveCommand command = new YtRemoveCommand(path, recursive, force);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void copy(YtPath sourcePath, YtPath destinationPath, boolean recursive) throws YtException {
        if (!Objects.equals(sourcePath.getCluster(), destinationPath.getCluster())) {
            throw new YtException("Unable to copy data between clusters from " + sourcePath + " to " + destinationPath);
        }

        YtCopyCommand command = new YtCopyCommand(sourcePath, destinationPath);
        command.setRecursive(recursive);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(destinationPath.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void move(YtPath sourcePath, YtPath destinationPath, boolean recursive, boolean force) throws YtException {
        if (!Objects.equals(sourcePath.getCluster(), destinationPath.getCluster())) {
            throw new YtException("Unable to move between clusters from " + sourcePath + " to " + destinationPath);
        }

        YtMoveCommand command = new YtMoveCommand(sourcePath, destinationPath, recursive, force);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(destinationPath.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void link(YtPath targetPath, YtPath linkPath, boolean force) throws YtException {
        if (!Objects.equals(targetPath.getCluster(), linkPath.getCluster())) {
            throw new YtException("Unable to link between clusters from " + targetPath + " to " + linkPath);
        }

        YtLinkCommand command = new YtLinkCommand(targetPath, linkPath, force);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(targetPath.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void set(YtPath path, JsonNode data) throws YtException {
        YtSetCommand command = new YtSetCommand(path, data);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public YtOperationId sort(YtPath sourcePaths, YtPath destinationPath, String... sortBy) throws YtException {
        return sort(Collections.singletonList(sourcePaths), destinationPath, sortBy);
    }

    @Override
    public YtOperationId sort(YtPath sourcePaths, YtPath destinationPath, List<String> sortBy) throws YtException {
        return sort(Collections.singletonList(sourcePaths), destinationPath, sortBy);
    }

    @Override
    public YtOperationId sort(List<YtPath> sourcePaths, YtPath destinationPath, String... sortBy) throws YtException {
        return sort(sourcePaths, destinationPath, Arrays.asList(sortBy));
    }

    @Override
    public YtOperationId sort(List<YtPath> sourcePaths, YtPath destinationPath, List<String> sortBy) throws YtException {
        if (sourcePaths.isEmpty()) {
            throw new IllegalArgumentException("Source paths is empty");
        }
        if (CollectionUtils.isEmpty(sortBy)) {
            throw new IllegalArgumentException("Sort by is empty");
        }
        if (!Objects.equals(sourcePaths.get(0).getCluster(), destinationPath.getCluster())) {
            throw new YtException("Unable to sort between clusters from " + sourcePaths.get(0).getCluster() + " to "
                    + destinationPath);
        }
        if (!exists(destinationPath)) {
            create(destinationPath, YtNode.NodeType.TABLE, true);
        }

        YtSortCommand command = new YtSortCommand(sourcePaths, destinationPath, sortBy);
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(destinationPath.getCluster()), command);

        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @Override
    public YtOperationId merge(YtPath sourceTable, YtPath destinationTable, String... mergeBy) throws YtException {
        return merge(Lists.newArrayList(sourceTable), destinationTable, mergeBy);
    }

    @Override
    public YtOperationId merge(List<YtPath> sourceTables, YtPath destinationTable, String... mergeBy) throws YtException {
        if (sourceTables.isEmpty()) {
            throw new IllegalArgumentException("Source tables is empty");
        }
        if (!Objects.equals(sourceTables.get(0).getCluster(), destinationTable.getCluster())) {
            throw new YtException("Unable to merge between clusters from " + sourceTables.get(0).getCluster() + " to "
                    + destinationTable);
        }

        YtMergeCommand command = new YtMergeCommand(sourceTables, destinationTable, mergeBy);
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(destinationTable.getCluster()), command);

        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @SuppressWarnings("Duplicates")
    @Override
    public YtOperationId map(YtMapCommand command) throws YtException {
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(command.getYtCluster()), command);
        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @SuppressWarnings("Duplicates")
    @Override
    public YtOperationId mapReduce(YtMapReduceCommand command) throws YtException {
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(command.getYtCluster()), command);
        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @SuppressWarnings("Duplicates")
    @Override
    public YtOperationId reduce(YtReduceCommand command) throws YtException {
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(command.getYtCluster()), command);
        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @Override
    public YtOperationId erase(YtPath table, Collection<YtTableRange> ranges) throws YtException {
        YtEraseCommand command = new YtEraseCommand(table, ranges);
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(table.getCluster()), command);

        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @Override
    public YtOperationId startOperation(AbstractYtJob command) throws YtException {
        YtResult<YtOperationId> result = ytService.execute(transaction, ytService.getCluster(command.getYtCluster()), command);
        ytService.throwIfError(result, command);
        YtOperationId operationId = result.getResult();
        log.info("{} started", operationId);
        return operationId;
    }

    @Override
    public YtOperation getOperation(YtOperationId operationId) throws YtException {
        YtGetOperationCommand command = new YtGetOperationCommand(operationId);
        YtResult<JsonNode> result = ytService.execute(transaction, ytService.getCluster(operationId.getCluster()), command);
        ytService.throwIfError(result, command);
        return createOperation(operationId, result.getResult());
    }

    @Override
    public void abortOperation(YtOperationId operationId) throws YtException {
        YtAbortOperationCommand command = new YtAbortOperationCommand(operationId);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(operationId.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public void completeOperation(YtOperationId operationId) throws YtException {
        YtCompleteOperationCommand command = new YtCompleteOperationCommand(operationId);
        YtResult<Void> result = ytService.execute(transaction, ytService.getCluster(operationId.getCluster()), command);
        ytService.throwIfError(result, command);
    }

    @Override
    public boolean waitFor(YtOperationId operationId) throws YtException {
        YtOperation operation;
        do {
            try {
                Thread.sleep(TimeUnit.SECONDS.toMillis(10));
            } catch (InterruptedException e) {
                throw new YtException("Interrupted while waiting for operation end", e);
            }
            operation = getOperation(operationId);
            log.info("{} progress: {}({}) {}/{}",
                    operation,
                    operation.getState(),
                    operation.getRunningJobs(),
                    operation.getCompletedJobs(),
                    operation.getTotalJobs());
        } while (!operation.getState().isTerminal());
        return operation.getState() == YtOperation.OperationState.COMPLETED;
    }

    private static YtOperation createOperation(YtOperationId operationId, JsonNode node) {
        String stateStr = node.get(ATTRIBUTE_OPERATION_STATE).textValue();
        String startTimeStr = node.get(ATTRIBUTE_OPERATION_START_TIME).textValue();
        JsonNode progressNode = node.get(ATTRIBUTE_OPERATION_PROGRESS);
        int totalJobs = 0;
        int pendingJobs = 0;
        int completedJobs = 0;
        int runningJobs = 0;
        if (progressNode != null) {
            JsonNode jobsNode = progressNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS);
            if (jobsNode != null) {
                JsonNode completedNode = jobsNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS_COMPLETED);
                completedJobs = completedNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS_COMPLETED_NON_INTERRUPTED).intValue();
                pendingJobs = jobsNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS_PENDING).intValue();
                runningJobs = jobsNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS_RUNNING).intValue();
                totalJobs = jobsNode.get(ATTRIBUTE_OPERATION_PROGRESS_JOBS_TOTAL).intValue();
            }
        }
        return new YtOperation(operationId,
                YtOperation.OperationState.R.valueOfOrUnknown(stateStr),
                new DateTime(startTimeStr),
                totalJobs, pendingJobs, runningJobs, completedJobs
        );
    }

    @Override
    public <E extends Exception> void writeTable(YtPath path, ThrowingConsumer<TableWriter, E> writerConsumer) throws YtException, E {
        YtTableData ytTableData = null;
        try {
            ytTableData = ytService.prepareTableData(path.getName(), writerConsumer);
            writeTable(path, ytTableData);
        } finally {
            if (ytTableData != null) {
                ytTableData.delete();
            }
        }
    }

    @Override
    public void writeTable(YtPath path, YtTableData tableData) throws YtException {
        writeTable(path, tableData, false);
    }

    @Override
    public void writeTable(YtPath path, YtTableData tableData, boolean append) throws YtException {
        if (!exists(path)) {
            create(path, YtNode.NodeType.TABLE, false);
        }

        YtWriteTableCommand command = new YtWriteTableCommand(path, tableData.getTableDataFile(), append);
        ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
    }

    YtReadAndSaveTableCommand.TableCache readAndSaveTable(YtPath path, YtTableRange range, boolean compress,
                                                          OutputFormat outputFormat, File saveToFile) throws YtException {
        YtReadAndSaveTableCommand command =
                new YtReadAndSaveTableCommand(path, range, compress, outputFormat, saveToFile);
        YtResult<YtReadAndSaveTableCommand.TableCache> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public List<JsonNode> selectRows(YtPath path, String where, String... fields) throws YtException {
        YtSelectRowsCommand command = new YtSelectRowsCommand(path, where, fields);
        YtResult<List<JsonNode>> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public <T> List<T> getRows(YtPath path, List<Object> keyValues, Class<T> rowClass) throws YtException {
        var tableReadDriver = YtTableReadDriver.createYSONDriver(rowClass);
        var command = new YtGetRowsCommand<T>(path, keyValues, tableReadDriver);
        YtResult<List<T>> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public <T> List<T> getRows(YtPath path, List<Object> keyValues, YtRowMapper<T> rowMapper) throws YtException {
        var tableReadDriver = YtTableReadDriver.createDSVDriver(rowMapper, YtMissingValueMode.SKIP_ROW);
        var command = new YtGetRowsCommand<T>(path, keyValues, tableReadDriver);
        YtResult<List<T>> result = ytService.execute(transaction, ytService.getCluster(path.getCluster()), command);
        ytService.throwIfError(result, command);
        return result.getResult();
    }

    @Override
    public void lock(YtPath path, YtLockMode mode) throws YtException {
        ytService.lock(transaction, path, mode);
    }

    public static class TableWriterImpl implements TableWriter, Closeable {
        private final OutputStream os;
        private final Map<String, Object> values = new HashMap<>();

        public TableWriterImpl(OutputStream os) {
            this.os = os;
        }

        @Override
        public TableWriter column(String name, Short value) {
            values.put(name, value);
            return this;
        }

        @Override
        public TableWriter column(String name, Integer value) {
            values.put(name, value);
            return this;
        }

        @Override
        public TableWriter column(String name, Long value) {
            values.put(name, value);
            return this;
        }

        @Override
        public TableWriter column(String name, Boolean value) {
            values.put(name, value);
            return this;
        }

        @Override
        public TableWriter column(String name, String value) {
            values.put(name, value);
            return this;
        }

        @Override
        public TableWriter columnObject(String name, Object object) {
            values.put(name, object);
            return this;
        }

        @Override
        public TableWriter column(String name, byte[] bytes) {
            return columnObject(name, bytes);
        }

        @Override
        public TableWriter rowEnd() throws YtException {
            try {
                write();
            } catch (IOException e) {
                throw new YtException("Unable to write row", e);
            }
            return this;
        }

        private void write() throws IOException {
            YtService.YSON_OM.writeValue(os, values);
            values.clear();
        }

        @Override
        public void close() throws IOException {
            if (!values.isEmpty()) {
                write();
            }
            os.close();
        }
    }
}
