package ru.yandex.webmaster3.storage.util.yt;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author avhaliullin
 */
public class YtJobSpec {
    private static final long DEFAULT_MEMORY_LIMIT = 512L * 1024 * 1024;

    private final Format inputFormat;
    private final Format outputFormat;
    private final List<YtOperationFilePathSpec> filePaths;
    private final String command;
    private final Object memoryLimit;
    private final Map<String, String> environment;
    private final Integer jobCount;

    YtJobSpec(List<YtOperationFilePathSpec> filePaths, String command, Format inputFormat, Format outputFormat, Long memoryLimit,
              Map<String, String> environment, Integer jobCount) {
        this.filePaths = filePaths;
        this.command = command;
        this.inputFormat = inputFormat;
        this.outputFormat = outputFormat;
        this.memoryLimit = Collections.singletonMap("$value", memoryLimit == null ? DEFAULT_MEMORY_LIMIT : memoryLimit);
        this.environment = environment;
        this.jobCount = jobCount;
    }

    @JsonProperty("output_format")
    public Object getOutputFormat() {
        return outputFormat.getFormatYson();
    }

    @JsonProperty("input_format")
    public Object getInputFormat() {
        return inputFormat.getFormatYson();
    }

    @JsonProperty("file_paths")
    public List<YtOperationFilePathSpec> getFilePaths() {
        return filePaths;
    }

    @JsonProperty("memory_limit")
    public Object getMemoryLimit() {
        return memoryLimit;
    }

    public String getCommand() {
        return command;
    }

    public Map<String, String> getEnvironment() {
        return environment;
    }

    @JsonProperty("job_count")
    public Integer getJobCount() {
        return jobCount;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {

        private final List<YtOperationFilePathSpec> filePaths = new ArrayList<>();
        private String command;
        private Format inputFormat = Format.createBinaryYson();
        private Format outputFormat = Format.createBinaryYson();
        private Long memoryLimit;
        private Map<String, String> environment = new HashMap<>();
        private Integer jobCount;

        public Builder setCommand(String command) {
            this.command = command;
            return this;
        }

        public Builder setCommand(YtOperationFilePathSpec commandFile, List<String> args) {
            this.command = "./" + commandFile.getName();
            filePaths.add(commandFile);
            if (!args.isEmpty()) {
                this.command += " " + String.join(" ", args);
            }
            return this;
        }

        public Builder setCommand(YtOperationFilePathSpec commandFile, String... args) {
            return setCommand(commandFile, Arrays.asList(args));
        }

        public Builder setInputFormat(Format inputFormat) {
            this.inputFormat = inputFormat;
            return this;
        }

        public Builder setOutputFormat(Format outputFormat) {
            this.outputFormat = outputFormat;
            return this;
        }

        public Builder setMemoryLimit(Long memoryLimit) {
            this.memoryLimit = memoryLimit;
            return this;
        }

        public Builder addEnvironmentVariable(String name, String value) {
            this.environment.put(name, value);
            return this;
        }

        public Builder setJobCount(Integer jobCount) {
            this.jobCount = jobCount;
            return this;
        }

        public YtJobSpec build() {
            if (StringUtils.isEmpty(command)) {
                throw new IllegalStateException("command is empty");
            }
            if (inputFormat == null) {
                throw new IllegalStateException("input_format is empty");
            }
            if (outputFormat == null) {
                throw new IllegalStateException("output_format is empty");
            }
            return new YtJobSpec(filePaths, command, inputFormat, outputFormat, memoryLimit, environment, jobCount);
        }
    }

//    private static final Map<String, Object> IO_FORMAT_BINARY_YSON;
//
//    static {
//        IO_FORMAT_BINARY_YSON = new HashMap<>();
//        Map<String, String> attrs = new HashMap<>();
//        attrs.put("format", "binary");
//        IO_FORMAT_BINARY_YSON.put("$attributes", attrs);
//        IO_FORMAT_BINARY_YSON.put("$value", "yson");
//    }
}
