package ru.yandex.webmaster3.storage.util.yt;

import java.util.*;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.fasterxml.jackson.databind.ser.std.ToStringSerializer;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Range;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

/**
 * @author aherman
 */
@JsonSerialize(using = ToStringSerializer.class)
public class YtPath implements Comparable<YtPath> {
    private final String cluster;
    private final List<String> path;

    YtPath(String cluster, List<String> path) {
        this.cluster = cluster;
        this.path = path;
    }

    public String getCluster() {
        return cluster;
    }

    public boolean isRoot() {
        return path.isEmpty();
    }

    public String getName() {
        if (isRoot()) {
            return StringUtils.EMPTY;
        }
        return Iterables.getLast(path);
    }

    public YtPath getParent() {
        if (path.isEmpty()) {
            return this;
        }
        if (path.size() == 1) {
            return new YtPath(cluster, Collections.emptyList());
        }

        return new YtPath(cluster, path.subList(0, path.size() - 1));
    }

    public String getPathWithoutCluster() {
        return toYtPath();
    }

    public String toYtPath() {
        return "//" + String.join("/", path);
    }

    String toYtPath(YtTableRange range) {
        Optional<String> rangeString = range.renderInPath();
        return rangeString.map(s -> "//" + String.join("/", path)
                + "[" + s + "]").orElseGet(this::toYtPath);
    }

    public String toYtPath(List<Object> keyValues) {
        String keyValuesStr = keyValues.stream().
                map(k -> {
                    try {
                        return YtService.OM.writeValueAsString(k);
                    } catch (JsonProcessingException e) {
                        // ignore
                    }

                    return k.toString();
                })
                .collect(Collectors.joining(","));

        return keyValuesStr.isEmpty()? toYtPath() : toYtPath() + "[(" + keyValuesStr + ")]";
    }

    public String toYtPath(Collection<YtTableRange> ranges) {
        String rangesStr = ranges.stream().
                map(YtTableRange::renderInPath)
                .flatMap(Optional::stream)
                .collect(Collectors.joining(","));
        return rangesStr.isEmpty()? toYtPath() : toYtPath() + "[" + rangesStr + "]";
    }

    String toYtPath(Range<Long> range) {
        if (!range.hasUpperBound() && !range.hasLowerBound()) {
            return toYtPath();
        }

        String result = "//" + String.join("/", path);
        result += '[';
        if (range.hasLowerBound()) {
            result += "#" + range.lowerEndpoint();
        }
        result += ':';
        if (range.hasUpperBound()) {
            result += "#" + range.upperEndpoint();
        }
        result += ']';
        return result;
    }

    public String toYqlPath() {
        StringBuilder yqlPathBuilder = new StringBuilder();
        if (cluster != null) {
            yqlPathBuilder.append(cluster).append(".");
        }
        yqlPathBuilder.append("`").append(toYtPath()).append("`");
        return yqlPathBuilder.toString();
    }


    @Override
    public String toString() {
        return cluster + ":" + "//" + String.join("/", path);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        YtPath ytPath = (YtPath) o;

        if (!cluster.equals(ytPath.cluster)) {
            return false;
        }
        return path.equals(ytPath.path);

    }

    @Override
    public int compareTo(@NotNull YtPath o) {
        return toString().compareTo(o.toString());
    }

    @Override
    public int hashCode() {
        int result = cluster.hashCode();
        result = 31 * result + path.hashCode();
        return result;
    }

    public static YtPath path(YtPath parent, String path) {
        if (path.startsWith("//")) {
            return create(parent.cluster, path);
        }

        ArrayList<String> newPath = new ArrayList<>(parent.path);
        if (StringUtils.contains(path, '/')) {
            String[] parts = StringUtils.split(path, '/');
            for (String part : parts) {
                checkPart(parent, part);
            }

            newPath.addAll(Arrays.asList(parts));
        } else {
            checkPart(parent, path);
            newPath.add(path);
        }
        return new YtPath(parent.cluster, newPath);
    }

    public static YtPath attribute(YtPath parent, String attrName) {
        return path(parent, "@" + attrName);
    }

    private static void checkPart(YtPath parent, String path) {
        if (".".equals(path) || "..".equals(path)) {
            throw new IllegalArgumentException("Path globbing not supported: " + parent + " " + path);
        }
    }

    @JsonCreator
    public static YtPath fromString(String path) {
        if (Strings.isNullOrEmpty(path)) {
            return null;
        }
        int colon = path.indexOf(':');
        if (colon <= 0) {
            throw new IllegalArgumentException("Invalid YT path: " + path);
        }
        String cluster = path.substring(0, colon);
        String pathPart = StringUtils.substring(path, colon + 1);
        String[] parts = parsePath(cluster, pathPart);
        return new YtPath(cluster, Arrays.asList(parts));
    }

    @NotNull
    private static String[] parsePath(String cluster, String pathPart) {
        if (!pathPart.startsWith("//")) {
            throw new IllegalArgumentException("Invalid YT path: " + cluster + ":" + pathPart);
        }
        pathPart = StringUtils.substring(pathPart, 2);
        if (pathPart.startsWith("/")) {
            throw new IllegalArgumentException("Invalid YT path: " + cluster + ":" + pathPart);
        }
        String[] parts = StringUtils.split(pathPart, '/');
        if (parts.length == 0) {
            throw new IllegalArgumentException("Invalid YT path: " + cluster + ":" + pathPart);
        }
        return parts;
    }

    public static YtPath create(String cluster, String path) {
        return new YtPath(cluster, Arrays.asList(parsePath(cluster, path)));
    }
}