package ru.yandex.webmaster3.storage.util.yt;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.RetryUtils;

import javax.annotation.Nullable;
import java.io.IOException;
import java.util.*;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import static com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_SINGLE_QUOTES;

/**
 * @author avhaliullin
 */
public class YtUtils {
    private static final Logger log = LoggerFactory.getLogger(YtUtils.class);
    private static final List<String> REDUCE_BY_V2 = Arrays.asList("TableId", "PartitionKey");
    private static final List<String> SORT_BY_V2 = Arrays.asList("TableId", "PartitionKey", "OrderKey");

    public static void recreateTables(YtCypressService cypressService, Collection<YtPath> tables, YtNodeAttributes attrs)
            throws YtException {
        for (YtPath path : tables) {
            if (cypressService.exists(path)) {
                cypressService.remove(path);
            }
            cypressService.create(path, YtNode.NodeType.TABLE, true, attrs);
        }
    }

    public static PrepareTablesForImportBuilder newPrepareTablesForImportBuilder() {
        return new PrepareTablesForImportBuilder();
    }

    @Nullable
    public static YtPath getLatestTablePath(YtService ytService, YtPath ytDirPath, Pattern tableNamePattern) {
        List<YtPath> tables;
        try {
            tables = ytService.withoutTransactionQuery(cypressService -> cypressService.list(ytDirPath));
        } catch (Exception e) {
            throw new WebmasterException("Failed to obtain tables list",
                    new WebmasterErrorResponse.YTServiceErrorResponse(YtUtils.class, e));
        }

        Optional<YtPath> latestTable = tables.stream()
                .filter(p -> tableNamePattern.matcher(p.toYtPath()).matches())
                .max(Comparator.naturalOrder());

        return latestTable.orElse(null);
    }

    /**
     * Билдер для генерации типичного таска map-reduce по подготовке временных таблиц, которые впоследующуем
     * будут импортированы в ClickHouse скриптом yt-import.py
     */
    public static class PrepareTablesForImportBuilder extends AbstractYtJobBuilder<PrepareTablesForImportBuilder> {

        private static final YtReduceIOSpec YT_REDUCE_IO_SPEC = new YtReduceIOSpec(1024 * 1024 * 128);//128 Mb
        private static final YtMapIOSpec YT_MAP_IO_SPEC = new YtMapIOSpec(1024 * 1024 * 128);//128 Mb
        private static final Map<String, Object> YT_SORT_IO_SPEC = Map.of("control_attributes", Map.of("enable_key_switch", true));

        private static final List<String> REDUCE_BY = Arrays.asList("part_id", "line_id");
        private static final List<String> SORT_BY = Arrays.asList("part_id", "line_id", "row_id");

        private YtPath binary;
        private String task;
        private List<String> mapperArgs = new ArrayList<>();
        private List<String> reducerArgs = new ArrayList<>();
        private int lines;
        private Integer mapperTableCount;
        private Integer textsTableCount = 0;
        private Integer reducerTableCount;
        private Long mapperMemoryLimit;
        private Long reducerMemoryLimit;

        private PrepareTablesForImportBuilder() {
        }

        public PrepareTablesForImportBuilder setBinary(YtPath binary) {
            this.binary = binary;
            return this;
        }

        public PrepareTablesForImportBuilder addMapperArg(String key, String value) {
            mapperArgs.add(key);
            mapperArgs.add(value);
            return this;
        }

        public PrepareTablesForImportBuilder addReducerArg(String key, String value) {
            mapperArgs.add(key);
            mapperArgs.add(value);
            return this;
        }

        public PrepareTablesForImportBuilder setTask(String task) {
            this.task = task;
            return this;
        }

        public PrepareTablesForImportBuilder setLines(int lines) {
            this.lines = lines;
            return this;
        }

        public PrepareTablesForImportBuilder setMapperTableCount(Integer mapperTableCount) {
            this.mapperTableCount = mapperTableCount;
            return this;
        }

        public PrepareTablesForImportBuilder setTextsTableCount(Integer textsTableCount) {
            this.textsTableCount = textsTableCount;
            return this;
        }

        public PrepareTablesForImportBuilder setReducerTableCount(Integer reducerTableCount) {
            this.reducerTableCount = reducerTableCount;
            return this;
        }

        public PrepareTablesForImportBuilder setMapperMemoryLimit(Long mapperMemoryLimit) {
            this.mapperMemoryLimit = mapperMemoryLimit;
            return this;
        }

        public PrepareTablesForImportBuilder setReducerMemoryLimit(Long reducerMemoryLimit) {
            this.reducerMemoryLimit = reducerMemoryLimit;
            return this;
        }

        public YtMapReduceCommand build() {
            Preconditions.checkState(binary != null);
            Preconditions.checkState(!Strings.isNullOrEmpty(task));
            Preconditions.checkState(lines > 0);
            preBuild();
            List<String> mapperArgs = new ArrayList<>(this.mapperArgs);
            // стандартные для всех параметры
            mapperArgs.add("--task");
            mapperArgs.add(task);
            mapperArgs.add("--stage");
            mapperArgs.add("map");
            mapperArgs.add("--tables");
            mapperArgs.add(String.valueOf(mapperTableCount != null ? mapperTableCount : outputTables.size()));
            mapperArgs.add("--lines");
            mapperArgs.add(String.valueOf(lines));
            YtJobSpec mapperSpec = YtJobSpec.newBuilder()
                    .setCommand(new YtOperationFilePathSpec(binary.toYtPath(), true, "mapper"), mapperArgs)
                    .setMemoryLimit(mapperMemoryLimit).build();

            List<String> reducerArgs = new ArrayList<>(this.mapperArgs);
            // стандартные для всех параметры
            reducerArgs.add("--task");
            reducerArgs.add(task);
            reducerArgs.add("--stage");
            reducerArgs.add("reduce");
            reducerArgs.add("--tables");
            reducerArgs.add(String.valueOf(reducerTableCount != null ? reducerTableCount.toString() : outputTables.size()));
            YtJobSpec reducerSpec = YtJobSpec.newBuilder()
                    .setCommand(new YtOperationFilePathSpec(binary.toYtPath(), true, "mapper"), reducerArgs)
                    .setMemoryLimit(reducerMemoryLimit).build();

            Map<String, Object> spec = new HashMap<>();
            spec.put("reduce_job_io", YT_REDUCE_IO_SPEC);
            spec.put("map_job_io", YT_MAP_IO_SPEC);

            return new YtMapReduceCommand(ytCluster, inputTables, outputTables, secureVault, mapperSpec, reducerSpec, null, SORT_BY, REDUCE_BY, spec);
        }

        public YtMapReduceCommand buildV2() {
            Preconditions.checkState(binary != null);
            Preconditions.checkState(!Strings.isNullOrEmpty(task));
            preBuild();
            List<String> mapperArgs = new ArrayList<>(this.mapperArgs);
            // стандартные для всех параметры
            mapperArgs.add("--task");
            mapperArgs.add(task);
            mapperArgs.add("--stage");
            mapperArgs.add("map_v2");
            mapperArgs.add("--tables");
            mapperArgs.add(String.valueOf(mapperTableCount != null ? mapperTableCount : outputTables.size()));
            mapperArgs.add("--texts-tables");
            mapperArgs.add(String.valueOf(textsTableCount));
            YtJobSpec mapperSpec = YtJobSpec.newBuilder()
                    .setCommand(new YtOperationFilePathSpec(binary.toYtPath(), true, "mapper"), mapperArgs)
                    .setMemoryLimit(mapperMemoryLimit).build();

            List<String> reducerArgs = new ArrayList<>(this.mapperArgs);
            // стандартные для всех параметры
            reducerArgs.add("--task");
            reducerArgs.add(task);
            reducerArgs.add("--stage");
            reducerArgs.add("reduce_v2");
            reducerArgs.add("--tables");
            reducerArgs.add(String.valueOf(reducerTableCount != null ? reducerTableCount.toString() : outputTables.size()));
            YtJobSpec reducerSpec = YtJobSpec.newBuilder()
                    .setCommand(new YtOperationFilePathSpec(binary.toYtPath(), true, "mapper"), reducerArgs)
                    .setMemoryLimit(reducerMemoryLimit).build();

            List<String> combinerArgs = new ArrayList<>(this.mapperArgs);
            // стандартные для всех параметры
            combinerArgs.add("--task");
            combinerArgs.add(task);
            combinerArgs.add("--stage");
            combinerArgs.add("combine_v2");
            combinerArgs.add("--tables");
            combinerArgs.add("1");
            YtJobSpec combinerSpec = YtJobSpec.newBuilder()
                    .setCommand(new YtOperationFilePathSpec(binary.toYtPath(), true, "mapper"), combinerArgs)
                    .setMemoryLimit(reducerMemoryLimit).build();

            Map<String, Object> spec = new HashMap<>();
            spec.put("reduce_job_io", YT_REDUCE_IO_SPEC);
            spec.put("map_job_io", YT_MAP_IO_SPEC);
            spec.put("sort_job_io", YT_SORT_IO_SPEC);
            spec.put("data_size_per_sort_job", 4L * 1024L * 1024L * 1024L);

            return new YtMapReduceCommand(ytCluster, inputTables, outputTables, secureVault, mapperSpec, reducerSpec, combinerSpec,
                    SORT_BY_V2, REDUCE_BY_V2, spec);
        }

        @Override
        protected PrepareTablesForImportBuilder getSelf() {
            return this;
        }
    }

    public static class TransactionExecutor {
        private final YtService ytService;
        private final YtPath workDir;

        public TransactionExecutor(final YtService ytService,
                                   final YtPath workDir) {
            this.ytService = ytService;
            this.workDir = workDir;
        }

        public void execute(YtTransactionService.TransactionProcess process) throws YtException {
            ytService.inTransaction(workDir).execute(process);
        }
    }

    public static class TransactionWriterBuilder {
        private static final ObjectMapper OM = new ObjectMapper().configure(ALLOW_SINGLE_QUOTES, true);
        private YtNodeAttributes attributes = new YtNodeAttributes().setCompressionCodec("lz4");

        private final YtPath tablePath;
        private final YtTableData tableData;
        private RetryUtils.RetryPolicy retryPolicy;

        public TransactionWriterBuilder(final YtPath tablePath,
                                        final YtTableData tableData) {
            this.tablePath = tablePath;
            this.tableData = tableData;
        }

        public TransactionWriterBuilder withSchema(final String tableSchema) throws IOException {
            attributes.getAttributes().put("schema", OM.readTree(tableSchema));
            return this;
        }

        public TransactionWriterBuilder withRetry(final RetryUtils.RetryPolicy retryPolicy) {
            this.retryPolicy = retryPolicy;
            return this;
        }

        public YtTransactionService.TransactionProcess build() {
            if (retryPolicy == null) {
                return cypressService -> internalBuild(cypressService).get();
            } else {
                return cypressService -> {
                    RetryUtils.query(retryPolicy, () -> internalBuild(cypressService).get());
                    return true;
                };
            }
        }

        private Supplier<Boolean> internalBuild(YtCypressService cypressService) {
            return () -> {
                try {
                    YtUtils.recreateTables(cypressService, Collections.singletonList(tablePath), attributes);
                    cypressService.writeTable(tablePath, tableData);
                } catch (YtException e) {
                    throw new RuntimeException(e);
                }
                return true;
            };
        }
    }

}
