package ru.yandex.webmaster3.storage.util.yt.transfer;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.base.Charsets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.transfer.YtTransferManagerAddTaskCommand.ClickhouseCopyOptions;

/**
 * Created by Oleg Bazdyrev on 27/08/2018.
 */
@Slf4j
@Service
public class YtTransferManager {

    private static final String HEADER_PARAMS = "X-TM-Parameters";
    private static final int SOCKET_TIMEOUT = 30_000;
    private static final String MDB_CLICKHOUSE_CLUSTER = "mdb-clickhouse";

    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new JodaModule())
            .registerModule(new ParameterNamesModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(JsonGenerator.Feature.AUTO_CLOSE_TARGET)
            .disable(SerializationFeature.WRITE_NULL_MAP_VALUES)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private final CloseableHttpClient httpClient = HttpClients.custom()
            .setDefaultRequestConfig(RequestConfig.custom()
                    .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT)
                    .build())
            .setConnectionTimeToLive(1, TimeUnit.MINUTES)
            .build();

    @Value("${external.yt.service.hahn.token}")
    private String authToken;
    @Value("http://transfer-manager.yt.yandex.net/api/v1")
    private String serviceUrl;
    @Value("${webmaster3.storage.clickhouse.service.login}")
    private String clickhouseLogin;
    @Value("${webmaster3.storage.mdb.clickhouse.password}")
    private String clickhousePassword;
    @Value("${webmaster3.storage.cloud.passport.oauthToken}")
    private String passportOAuthToken;

    public String addTask(YtPath source, YtPath target) throws YtException {
        log.info("Adding transfer manager task for copy {} to {}", source, target);
        return addTask(YtTransferManagerAddTaskCommand.create(source, target));
    }

    public String addTask(YtTransferManagerAddTaskCommand command) throws YtException {
        return addTask(command, null);
    }

    public String addTask(YtTransferManagerAddTaskCommand command, YtTransferManagerMutation mutation) throws YtException {
        HttpPost post = new HttpPost(serviceUrl + "/tasks/");
        try {
            String data = OM.writeValueAsString(command);
            post.setEntity(new StringEntity(data, ContentType.APPLICATION_JSON));
            post.setHeader(HttpHeaders.AUTHORIZATION, "OAuth " + authToken);
            if (mutation != null) {
                post.setHeader(HEADER_PARAMS, OM.writeValueAsString(mutation));
            }
        } catch (JsonProcessingException e) {
            throw new YtException("Json error", e);
        }
        try (CloseableHttpResponse response = httpClient.execute(post)) {
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                return IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
            } else {
                String message = OM.readTree(response.getEntity().getContent()).get("message").asText();
                log.error("Transfer-manager returned code {}. Message: {}", statusCode, message);
                throw new YtException("Transfer-manager returned code " + statusCode);
            }
        } catch (IOException e) {
            log.error("Error executing request", e);
            throw new YtException("Cannot execute request", e);
        }
    }

    public YtTransferTaskInfo getTask(String id) throws YtException {
        log.info("Requesting transfer manager about task {}", id);
        HttpGet get = new HttpGet(serviceUrl + "/tasks/" + id + "/");
        get.setHeader(HttpHeaders.AUTHORIZATION, "OAuth " + authToken);
        try (CloseableHttpResponse response = httpClient.execute(get)) {
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode == HttpStatus.SC_OK) {
                return OM.readValue(response.getEntity().getContent(), YtTransferTaskInfo.class);
            } else if (statusCode == HttpStatus.SC_NOT_FOUND) {
                return null;
            } else {
                String message = OM.readTree(response.getEntity().getContent()).get("message").asText();
                log.error("Transfer-manager returned code {}. Message: {}", statusCode, message);
                throw new YtException("Transfer-manager returned code " + statusCode);
            }
        } catch (IOException e) {
            log.error("Error executing request", e);
            throw new YtException("Cannot execute request", e);
        }
    }

    public String copyFromYtToClickhouse(YtPath sourceTable, String destinationTable, String chClusterId,
                                         ClickhouseCopyOptions copyOptions, YtTransferManagerMutation mutation, ClickhouseCopyToolSettings settingsPatch) {
        var command = YtTransferManagerAddTaskCommand.builder()
                .sourceCluster(sourceTable.getCluster())
                .sourceTable(sourceTable.getPathWithoutCluster())
                .targetCluster(MDB_CLICKHOUSE_CLUSTER)
                .targetTable(destinationTable)
                .mdbAuth(YtTransferManagerAddTaskCommand.MdbAuth.builder().oauthToken(passportOAuthToken).build())
                .mdbClusterAddress(YtTransferManagerAddTaskCommand.MdbClusterAddress.builder().clusterId(chClusterId).build())
                .clickhouseCredentials(YtTransferManagerAddTaskCommand.ClickhouseCredentials.builder().user(clickhouseLogin).password(clickhousePassword).build())
                .clickhouseCopyOptions(copyOptions)
                .clickhouseCopyToolSettingsPatch(settingsPatch)
                .build();
        return addTask(command, mutation);
    }

    public YtTransferTaskInfo waitForTaskResult(String taskId) {
        while (true) {
            YtTransferTaskInfo taskInfo = getTask(taskId);
            if (taskInfo == null || taskInfo.getState().isFinal()) {
                return taskInfo;
            }
            try {
                Thread.sleep(30000L);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new YtException("Interrupted!", e);
            }
        }
    }

    public void waitForTaskCompletion(String taskId) {
        YtTransferTaskInfo taskInfo = waitForTaskResult(taskId);
        if (taskInfo == null) {
            throw new YtException("Transfer " + taskId + " was lost");
        }
        if (taskInfo.getState() != YtTransferTaskState.COMPLETED) {
            throw new YtException("Transfer " + taskId + " failed: " + taskInfo.getError());
        }
    }

}
