package ru.yandex.webmaster3.storage.util.zookeeper;

import org.apache.commons.lang3.tuple.Pair;
import org.apache.curator.drivers.AdvancedTracerDriver;
import org.apache.curator.drivers.EventTrace;
import org.apache.curator.drivers.OperationTrace;
import org.apache.zookeeper.KeeperException;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimer;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author avhaliullin
 */
public class ZookeeperTracerService {
    private static final Logger log = LoggerFactory.getLogger(ZookeeperTracerService.class);

    private static final List<String> OP_NAME_SUFFIXES = Arrays.asList("-Foreground", "-Background");
    private static final int MAX_DIMENSIONS = KeeperException.Code.values().length * 20;

    private final SolomonTraceDriver traceDriver = new SolomonTraceDriver();
    private final Map<KeeperException.Code, ConcurrentHashMap<Pair<String, String>, SolomonTimer>> code2NameAndPath2Timer;
    private final ConcurrentHashMap<Pair<String, String>, SolomonTimer> unknownCodeNameAndPath2Timer;

    public ZookeeperTracerService() {
        this.code2NameAndPath2Timer = new EnumMap<>(KeeperException.Code.class);
        for (KeeperException.Code code : KeeperException.Code.values()) {
            code2NameAndPath2Timer.put(code, new ConcurrentHashMap<>());
        }
        this.unknownCodeNameAndPath2Timer = new ConcurrentHashMap<>();
    }

    private SolomonMetricRegistry solomonMetricRegistry;
    private SolomonTimerConfiguration solomonTimerConfiguration;

    public AdvancedTracerDriver getTraceDriver() {
        return traceDriver;
    }

    private SolomonTimer createTimer(KeeperException.Code code, String name, String path) {
        return solomonMetricRegistry.createTimer(
                solomonTimerConfiguration,
                SolomonKey.create("operation", name)
                        .withLabel("path_prefix", path)
                        .withLabel("result_code", code == null ? "<unknown>" : code.name())
        );
    }

    private SolomonTimer getTimer(KeeperException.Code code, String name, String path) {
        ConcurrentHashMap<Pair<String, String>, SolomonTimer> map = code == null
                ? unknownCodeNameAndPath2Timer
                : code2NameAndPath2Timer.get(code);
        boolean overflow = map.size() > MAX_DIMENSIONS;
        String finalPath = overflow ? "<overflow>" : path;
        if (overflow) {
            log.error("Too many distinct zookeeper paths, use " + finalPath + " instead of " + path);
        }
        return map.computeIfAbsent(Pair.of(name, path), ign -> createTimer(code, name, finalPath));
    }

    private static String getPathPrefix(String path) {
        if (path == null) {
            return "";
        }
        String[] parts = path.split("/");
        return parts[1];
    }

    private static String normalizeOpName(String name) {
        for (String suffix : OP_NAME_SUFFIXES) {
            if (name.endsWith(suffix)) {
                name = name.substring(0, name.length() - suffix.length());
            }
        }
        return name;
    }

    private class SolomonTraceDriver extends AdvancedTracerDriver {
        @Override
        public void addTrace(OperationTrace trace) {
            if (trace.getReturnCode() != KeeperException.Code.OK.intValue()) {
                log.info("ZookeeperOperation {} path={} code={} timeMs={} reqBytes={} respBytes={}",
                        trace.getName(),
                        trace.getPath(),
                        trace.getReturnCode(),
                        trace.getLatencyMs(),
                        trace.getRequestBytesLength(),
                        trace.getResponseBytesLength()
                );
            }
            getTimer(
                    KeeperException.Code.get(trace.getReturnCode()),
                    normalizeOpName(trace.getName()),
                    getPathPrefix(trace.getPath())
            ).update(Duration.millis(trace.getLatencyMs()));
        }

        @Override
        public void addEvent(EventTrace trace) {
            log.info("ZookeeperEvent {} session {}", trace.getName(), trace.getSessionId());
        }
    }

    @Required
    public void setSolomonMetricRegistry(SolomonMetricRegistry solomonMetricRegistry) {
        this.solomonMetricRegistry = solomonMetricRegistry;
    }

    @Required
    public void setSolomonTimerConfiguration(SolomonTimerConfiguration solomonTimerConfiguration) {
        this.solomonTimerConfiguration = solomonTimerConfiguration;
    }
}
