package ru.yandex.webmaster3.storage.vanadium;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Range;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;

/**
 * Created by Oleg Bazdyrev on 10/12/2020.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class VanadiumService extends AbstractExternalAPIService {

    private static final String METHOD_ENQUEUE = "batch/enqueue";
    private static final String METHOD_STATUS = "batch/status";
    //private static final String QUEUE_NAME = "main";
    private static final String USER_NAME = "robot-webmaster";
    private static final String BATCH_PREFIX_COMMON = "webmaster-common-";
    private static final String TASK_PREFIX = "task-";
    private static final String ATTR_BATCH_INFO = "batch-info";
    private static final String ATTR_TELLURIUM_STATUS = "tellurium.status";
    private static final List<String> YT_CLUSTERS = Lists.newArrayList("hahn", "arnold");
    private static final int SOCKET_TIMEOUT = (int) TimeUnit.SECONDS.toMillis(900);
    private static final long MIN_TABLE_AGE_FOR_REMOVAL = TimeUnit.DAYS.toMillis(3L);

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> URL = SCHEMA.addColumn("url", YtColumn.Type.STRING);
        YtColumn<Long> WIDTH = SCHEMA.addColumn("view-width", YtColumn.Type.INT_64);
        YtColumn<Long> HEIGHT = SCHEMA.addColumn("view-height", YtColumn.Type.INT_64);
        YtColumn<String> PLATFORM = SCHEMA.addColumn("platform-type", YtColumn.Type.STRING);
    }

    private final YtService ytService;

    @Value("${webmaster3.storage.vanadium.apiUrl}")
    private String apiUrl;
    @Value("${webmaster3.storage.vanadium.workDir.path}")
    private YtPath workDir;
    @Value("${webmaster3.storage.vanadium.mainQueue.path}")
    private YtPath mainQueueDir;

    private CloseableHttpClient httpClient;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig).build();
    }

    /**
     * Запускает таску Теллуриума для скриншота урлов из указанной таблицы (в таблице должна быть колонка url c искомыми данными)
     */
    public String startScreenshotsTask(YtPath urlsTable, int width, int height, Platform platform) {
        List<String> res = null;
        try {
            res = ytService.withoutTransactionQuery(cypressService -> {
                AsyncTableReader<Row> tableReader = new AsyncTableReader<>(cypressService, urlsTable, Range.all(),
                        YtTableReadDriver.createYSONDriver(Row.class));
                List<String> urls = new ArrayList<>();
                try (AsyncTableReader.TableIterator<Row> iterator = tableReader.read()) {
                    while (iterator.hasNext()) {
                        urls.add(iterator.next().getUrl());
                    }
                } catch (Exception e) {
                    throw new WebmasterException("Error reading urls table", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                }
                return urls;
            });
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new WebmasterException("Interrupted", new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Interrupted"), e);
        }
        return startScreenshotsTask(urlsTable.getCluster(), res, width, height, platform);
    }

    /**
     * Запускает таску Теллуриума для скриншота указанного списка
     */
    public String startScreenshotsTask(String cluster, Collection<String> urls, int width, int height, Platform platform) {
        // создаем табличку с нашими урлами
        // название таблички task-timeUUID
        UUID id = UUIDs.timeBased();
        YtPath taskTable = YtPath.path(YtPath.create(cluster, workDir.getPathWithoutCluster()), TASK_PREFIX + id.toString());
        ytService.inTransaction(cluster).execute(cypressService -> {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
            cypressService.create(taskTable, YtNode.NodeType.TABLE, true, attributes, true);
            cypressService.writeTable(taskTable, tableWriter -> {
                for (String url : urls) {
                    F.URL.set(tableWriter, url);
                    F.WIDTH.set(tableWriter, (long) width);
                    F.HEIGHT.set(tableWriter, (long) height);
                    F.PLATFORM.set(tableWriter, platform.name().toLowerCase());
                    tableWriter.rowEnd();
                }
            });
            return true;
        });
        // запускаем импорт
        VanadiumTaskInput input = VanadiumTaskInput.builder().ytCluster(taskTable.getCluster()).inputTable(taskTable.getPathWithoutCluster())
                .batchNamePrefix(BATCH_PREFIX_COMMON).user(USER_NAME).build();
        VanadiumTaskOutput output = enqueueBatch(input);
        // сохраним батч в атрибутах нашей таблички
        ytService.inTransaction(cluster).execute(cypressService -> {
            cypressService.set(YtPath.attribute(taskTable, ATTR_BATCH_INFO), JsonMapping.OM.valueToTree(output));
            return true;
        });
        return cluster + "-" + id.toString();
    }

    @ExternalDependencyMethod("batch-enqueue")
    private VanadiumTaskOutput enqueueBatch(VanadiumTaskInput input) {
        log.info("Enqueuing screenshots batch: {}", input);
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
            HttpPost post = new HttpPost(apiUrl + METHOD_ENQUEUE);
            post.setEntity(new StringEntity(JsonMapping.writeValueAsString(input), ContentType.APPLICATION_JSON));
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                String content = EntityUtils.toString(response.getEntity());
                if (response.getStatusLine().getStatusCode() / 100 == 2) {
                    VanadiumTaskOutput output = JsonMapping.readValue(content, VanadiumTaskOutput.class);
                    log.info("Successfully enqueued batch: {}", output);
                    return output;
                } else {
                    log.info("Error occurred when enqueuing batch. Status code: {}, status line: {}", response.getStatusLine().getStatusCode(), content);
                    throw new WebmasterException("Error occurred when enqueuing vanadium batch", new WebmasterErrorResponse.VanadiumErrorResponse(getClass()));
                }
            } catch (IOException e) {
                throw new WebmasterException("IO exception occurred", new WebmasterErrorResponse.VanadiumErrorResponse(getClass(), e), e);
            }
        });
    }

    @ExternalDependencyMethod("batch-status")
    private BatchStatusOutput getBatchStatus(String cluster, String batchName, String queueName) {
        return trackQuery(new JavaMethodWitness() {
                          }, ALL_ERRORS_INTERNAL, () -> {
            HttpPost post = new HttpPost(apiUrl + METHOD_STATUS);
            BatchStatusInput input = BatchStatusInput.builder().ytCluster(cluster).batchTableName(batchName).queueName(queueName).build();
            post.setEntity(new StringEntity(JsonMapping.writeValueAsString(input), ContentType.APPLICATION_JSON));
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                String content = EntityUtils.toString(response.getEntity());
                if (response.getStatusLine().getStatusCode() / 100 == 2) {
                    BatchStatusOutput output = JsonMapping.readValue(content, BatchStatusOutput.class);
                    log.info("Status: {}", output);
                    return output;
                } else {
                    log.info("Error occurred when requesting batch status. Status code: {}, status line: {}", response.getStatusLine().getStatusCode(), content);
                    throw new WebmasterException("Vanadium error: " + content, new WebmasterErrorResponse.VanadiumErrorResponse(getClass()));
                }
            } catch (IOException e) {
                throw new WebmasterException("IO exception occurred", new WebmasterErrorResponse.VanadiumErrorResponse(getClass(), e), e);
            }

        });
    }

    /**
     * Получение статуса таски Теллуриума
     */
    public BatchStatusOutput getBatchStatus(String taskId) {
        YtPath taskTable = taskTableFromTaskId(taskId);
        return ytService.inTransaction(taskTable).query(cypressService -> {
            YtNode node = cypressService.getNode(taskTable);
            VanadiumTaskOutput batchInfo = JsonMapping.readValue(node.getNodeMeta().get(ATTR_BATCH_INFO).traverse(), VanadiumTaskOutput.class);
            return getBatchStatus(taskTable.getCluster(), batchInfo.getBatchName(), batchInfo.getQueueName());
        });
    }

    private YtPath taskTableFromTaskId(String taskId) {
        int dashIdx = taskId.indexOf('-');
        Preconditions.checkArgument(dashIdx != -1, "Wrong taskId format");
        String cluster = taskId.substring(0, dashIdx);
        String id = taskId.substring(dashIdx + 1);
        YtPath taskTable = YtPath.path(YtPath.create(cluster, workDir.getPathWithoutCluster()), TASK_PREFIX + id);
        return taskTable;
    }

    @Scheduled(cron = "0 0 0/6 * * *")
    public void cleanOldTables() {
        long now = System.currentTimeMillis();
        for (String cluster : YT_CLUSTERS) {
            YtPath workDir = YtPath.create(cluster, this.workDir.getPathWithoutCluster());
            ytService.inTransaction(workDir).execute(cypressService -> {
                List<YtPath> tables = cypressService.list(workDir);
                for (YtPath table : tables) {
                    if (!table.getName().startsWith(TASK_PREFIX)) {
                        continue;
                    }
                    UUID id = UUID.fromString(table.getName().substring(TASK_PREFIX.length()));
                    long tableMs = UUIDs.unixTimestamp(id);
                    if ((now - tableMs) > MIN_TABLE_AGE_FOR_REMOVAL) {
                        cypressService.remove(table);
                    }
                }
                return true;
            });
        }
    }


    public enum Platform {
        TOUCH,
        DESKTOP,
    }

    public enum Status {
        UNKNOWN,
        @JsonProperty("new")
        NEW,
        @JsonProperty("progress")
        PROGRESS,
        @JsonProperty("succeeded")
        SUCCEEDED,
        @JsonProperty("failed")
        FAILED,
        @JsonProperty("invalid")
        INVALID,
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class Row {
        @JsonProperty("url")
        String url;
    }

    @lombok.Value
    @Builder
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static class VanadiumTaskInput {
        @JsonProperty("yt-cluster")
        String ytCluster;
        @JsonProperty("user")
        String user;
        @JsonProperty("input-table")
        String inputTable;
        @JsonProperty("batch-name-prefix")
        String batchNamePrefix;
    }

    @lombok.Value
    @Builder
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class VanadiumTaskOutput {
        @JsonProperty("batch-name")
        String batchName;
        @JsonProperty("batch-uuid")
        UUID batchUuid;
        @JsonProperty("queue-name")
        String queueName;
    }

    @lombok.Value
    @Builder
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static class BatchStatusInput {
        @JsonProperty("yt-cluster")
        String ytCluster;
        @JsonProperty("queue-name")
        String queueName;
        @JsonProperty("batch-table-name")
        String batchTableName;
    }

    @lombok.Value
    @Builder
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class BatchStatusOutput {
        @JsonProperty("yt-cluster")
        String ytCluster;
        @JsonProperty("queue-name")
        String queueName;
        @JsonProperty("batch-table-name")
        String batchTableName;
        @JsonProperty("batch-status")
        Status batchStatus;
        @JsonProperty("batch-output-table-path")
        String batchOutputTablePath;
    }

}
