package ru.yandex.webmaster3.storage.verification;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import com.google.common.annotations.VisibleForTesting;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.xbill.DNS.Type;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.IUserHostVerifier;
import ru.yandex.webmaster3.core.host.verification.UinUtil;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.DNSRecord;
import ru.yandex.webmaster3.core.host.verification.fail.DNSRecordNotFoundFailInfo;
import ru.yandex.webmaster3.storage.verification.dns.DnsLookupService;

/**
 * @author avhaliullin
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DNSVerifier implements IUserHostVerifier {
    private static final Logger log = LoggerFactory.getLogger(DNSVerifier.class);

    @VisibleForTesting
    static final String YANDEX_VERIFICATION_STRING = "yandex-verification";

    private final DnsLookupService dnsLookupService;
    private final List<String> nameServers;

    @Override
    public Optional<VerificationFailInfo> verify(long userId, WebmasterHostId hostId, UUID recordId, long verificationUin, VerificationCausedBy verificationCausedBy) {
        String uinString = UinUtil.getUinString(verificationUin);

        String hostName = hostId.getPunycodeHostname();
        log.info("DNS requesting: " + hostName);
        Pair<DnsLookupService.LookupStatus, List<DNSRecord>> lookupResult = dnsLookupService.lookup(nameServers, hostName, Type.TXT);
        if (lookupResult.getLeft() == DnsLookupService.LookupStatus.HOST_NOT_FOUND) {
            return Optional.of(new DNSRecordNotFoundFailInfo(false, Collections.emptyList(), Collections.emptyList()));
        }
        List<DNSRecord> records = lookupResult.getRight();

        List<DNSRecord> txtRecords = new ArrayList<>();
        List<DNSRecord> verificationRecords = new ArrayList<>();

        for (DNSRecord record : records) {
            log.debug("DNS record: " + record);
            String recordString = record.getData();
            log.debug("DNS: required content: " + uinString);
            log.debug("DNS: found content   : " + recordString);
            if (recordString.contains(YANDEX_VERIFICATION_STRING)) {
                verificationRecords.add(record);
                //Такая проверка слабее, чем можно было бы ожидать - оставляю ее для совместимости со старым ВМ
                if (recordString.contains(uinString)) {
                    return Optional.empty();
                }
            } else {
                txtRecords.add(record);
            }
        }
        return Optional.of(new DNSRecordNotFoundFailInfo(true, txtRecords, verificationRecords));
    }

    @Override
    public boolean isApplicable(WebmasterHostId hostId, long userId) {
        return true;
    }
}
