package ru.yandex.webmaster3.storage.verification;

import java.io.IOException;
import java.io.StringReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.EnumSet;
import java.util.Optional;
import java.util.UUID;

import lombok.Setter;
import org.cyberneko.html.parsers.SAXParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.IUserHostVerifier;
import ru.yandex.webmaster3.core.host.verification.PageUnavailableReason;
import ru.yandex.webmaster3.core.host.verification.UinUtil;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationHttpResponseInfo;
import ru.yandex.webmaster3.core.host.verification.fail.PageUnavailableFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.WrongHtmlPageContentFailInfo;
import ru.yandex.webmaster3.core.util.Either;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.ZoraResponseDocumentUtil;
import ru.yandex.webmaster3.core.zora.data.response.ZoraUrlFetchResponse;
import ru.yandex.wmtools.common.error.InternalException;

/**
 * @author avhaliullin
 */
public class HtmlFileVerifier implements IUserHostVerifier {
    private static final Logger log = LoggerFactory.getLogger(HtmlFileVerifier.class);

    @Setter
    private VerificationPageDownloadService verificationPageDownloadService;

    @Override
    public Optional<VerificationFailInfo> verify(long userId, WebmasterHostId hostId, UUID recordId, long verificationUin, VerificationCausedBy verificationCausedBy) {
        URL urlAddress = getVerificationUrl(hostId, verificationUin);

        Either<ZoraUrlFetchResponse, VerificationFailInfo> downloadResult =
                verificationPageDownloadService.getVerificationPage(
                        urlAddress,
                        1,
                        EnumSet.of(
                                RedirectResolveFeature.ALLOW_RELATIVE,
                                RedirectResolveFeature.ALLOW_SWITCH_PORT,
                                RedirectResolveFeature.ALLOW_SWITCH_SCHEMA,
                                RedirectResolveFeature.ALLOW_SWITCH_WWW)
                );
        if (downloadResult.isLeft()) {
            ZoraUrlFetchResponse zoraUrlFetchResponse = downloadResult.getLeftUnsafe();
            return doVerify(zoraUrlFetchResponse, verificationUin, urlAddress);
        } else {
            return downloadResult.getRight();
        }
    }

    public static URL getVerificationUrl(WebmasterHostId hostId, long verificationUin) {
        try {
            return new URL(IdUtils.hostIdToUrl(hostId) + "/yandex_" + UinUtil.getUinString(verificationUin) + ".html");
        } catch (MalformedURLException e) {
            throw new RuntimeException("Unexpected error: failed to build html file verification url for host " + hostId + " uin " + verificationUin);
        }
    }

    private Optional<VerificationFailInfo> doVerify(ZoraUrlFetchResponse zoraUrlFetchResponse, long verificationUin, URL urlAddress) {
        String expectedContent = "Verification: " + UinUtil.getUinString(verificationUin);
        VerifyingSAXHandler verifyingSAXHandler = new VerifyingSAXHandler(expectedContent);
        try {

            SAXParser saxParser = new SAXParser();
            saxParser.setFeature("http://cyberneko.org/html/features/balance-tags", false);
            saxParser.setContentHandler(verifyingSAXHandler);
            saxParser.setErrorHandler(verifyingSAXHandler);
            saxParser.parse(new InputSource(new StringReader(ZoraResponseDocumentUtil.getResponseString(zoraUrlFetchResponse))));
            if (verifyingSAXHandler.isVerified()) {
                return Optional.empty();
            } else {
                VerificationHttpResponseInfo responseInfo = VerificationHttpResponseInfo.createFromZoraResponse(zoraUrlFetchResponse);
                return Optional.of(new WrongHtmlPageContentFailInfo(responseInfo));
            }
        } catch (SAXException e) {
            if (verifyingSAXHandler.isVerified()) {
                return Optional.empty();
            } else {
                log.warn("Checking html-file: Not verified: IOException while reading file: " + urlAddress, e);
                VerificationHttpResponseInfo responseInfo = VerificationHttpResponseInfo.createFromZoraResponse(zoraUrlFetchResponse);
                return Optional.of(new WrongHtmlPageContentFailInfo(responseInfo));
            }
        } catch (InternalException | IOException e) {
            if (verifyingSAXHandler.isVerified()) {
                return Optional.empty();
            } else {
                log.warn("Checking html-file: Not verified: IOException while reading file with meta-tag: " + urlAddress, e);
                VerificationHttpResponseInfo responseInfo = VerificationHttpResponseInfo.createFromZoraResponse(zoraUrlFetchResponse);
                return Optional.of(
                        new PageUnavailableFailInfo(
                                urlAddress,
                                responseInfo,
                                PageUnavailableReason.CONTENT_PROCESSING_ERROR
                        )
                );
            }
        }
    }

    @Override
    public boolean isApplicable(WebmasterHostId hostId, long userId) {
        return true;
    }

    private static class VerifyingSAXHandler extends DefaultHandler {
        private boolean found = false;
        private String faultLog = null;
        private boolean insideBody = false;
        private boolean foundBody = false;

        private final StringBuilder foundContent = new StringBuilder();
        private final String requiredContent;

        public VerifyingSAXHandler(String content) {
            this.requiredContent = content;
        }

        public boolean isVerified() {
            return found && (faultLog == null);
        }

        public String getFaultLog() {
            return faultLog;
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes atts) {
            if (faultLog != null) {
                return;
            }

            if (insideBody) {
                faultLog = "Unexpected tag <" + localName + "> inside <body>";
                return;
            }

            if ("body".equalsIgnoreCase(localName)) {
                if (foundBody) {
                    faultLog = "Second <body> tag found in file, while expected only one";
                    return;
                }

                insideBody = true;
                foundBody = true;
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) {
            if (faultLog != null) {
                return;
            }

            if (!insideBody) {
                return;
            }

            if ("body".equalsIgnoreCase(localName)) {
                insideBody = false;
            } else {
                faultLog = "Unexpected tag </" + localName + "> inside <body>";
            }
        }

        @Override
        public void characters(char ch[], int start, int length) {
            if (faultLog != null) {
                return;
            }

            if (!insideBody) {
                return;
            }

            foundContent.append(ch, start, length);

            if (!requiredContent.startsWith(foundContent.toString().trim())) {
                faultLog = "Expected '" + requiredContent + "', but found '" + foundContent + "' inside <body>";
                return;
            }

            if (requiredContent.equals(foundContent.toString().trim())) {
                found = true;
            }

        }
    }
}
