package ru.yandex.webmaster3.storage.verification;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.xbill.DNS.Type;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.IUserHostVerifier;
import ru.yandex.webmaster3.core.host.verification.PageUnavailableReason;
import ru.yandex.webmaster3.core.host.verification.UinUtil;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationHttpResponseInfo;
import ru.yandex.webmaster3.core.host.verification.fail.DNSRecord;
import ru.yandex.webmaster3.core.host.verification.fail.PDDEmuVerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.PageUnavailableFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.WrongHtmlPageContentFailInfo;
import ru.yandex.webmaster3.core.util.Either;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.ZoraResponseDocumentUtil;
import ru.yandex.webmaster3.core.zora.data.response.ZoraUrlFetchResponse;
import ru.yandex.webmaster3.storage.verification.dns.DnsLookupService;
import ru.yandex.wmtools.common.error.InternalException;

import static ru.yandex.webmaster3.core.host.verification.fail.PDDEmuVerificationFailInfo.DNSRecordCNAMENotFoundFailInfo;

/**
 * @author leonidrom
 * <p>
 * WMC-6592. Реализует логику верификации как в закопанном ПДД
 */
public class PDDEmuVerifier implements IUserHostVerifier {
    private static final Logger log = LoggerFactory.getLogger(PDDEmuVerifier.class);
    private static final String VERIFICATION_PREFIX = "yamail-";
    private static final Set<String> ALLOWED_CNAMES = Set.of("mail.yandex.ru.", "mail.yandex.com.", "mail.yandex.tr.");

    private final DnsLookupService dnsLookupService;
    private final VerificationPageDownloadService verificationPageDownloadService;

    @Autowired
    public PDDEmuVerifier(
            VerificationPageDownloadService verificationPageDownloadService,
            DnsLookupService dnsLookupService) {
        this.verificationPageDownloadService = verificationPageDownloadService;
        this.dnsLookupService = dnsLookupService;
    }

    @Override
    public Optional<VerificationFailInfo> verify(long userId, WebmasterHostId hostId, UUID recordId, long verificationUin, VerificationCausedBy verificationCausedBy) {
        log.info("Trying to verify host {} for user {} with PDD_EMU", userId, hostId);
        var dnsFailInfoOpt = tryVerifyDNS(hostId, verificationUin);
        if (dnsFailInfoOpt.isEmpty()) {
            log.info("Verified host {} for user {} with PDD_EMU using CNAME", userId, hostId);
            return Optional.empty();
        }

        var verifyHtmlFileFailOpt = tryVerifyHtmlFile(hostId, verificationUin, verificationCausedBy);
        if (verifyHtmlFileFailOpt.isEmpty()) {
            log.info("Verified host {} for user {} with PDD_EMU using HTML file", userId, hostId);
            return Optional.empty();
        }

        log.info("Failed to verify host {} for user {} with PDD_EMU", userId, hostId);
        return Optional.of(new PDDEmuVerificationFailInfo(dnsFailInfoOpt.get(), verifyHtmlFileFailOpt.get()));
    }

    @Override
    public boolean isApplicable(WebmasterHostId hostId, long userId) {
        return true;
    }

    private Optional<DNSRecordCNAMENotFoundFailInfo> tryVerifyDNS(WebmasterHostId hostId, long verificationUin) {
        String secret = UinUtil.getUinString(verificationUin);
        String lookupName = VERIFICATION_PREFIX + secret + "." + hostId.getPunycodeHostname();

        log.info("DNS requesting CNAME: {}", lookupName);
        Pair<DnsLookupService.LookupStatus, List<DNSRecord>> lookupResult = dnsLookupService.lookup(lookupName, Type.CNAME);
        if (lookupResult.getLeft() == DnsLookupService.LookupStatus.HOST_NOT_FOUND) {
            log.info("DNS CNAME {} is not found", lookupName);
            return Optional.of(new DNSRecordCNAMENotFoundFailInfo(false, Collections.emptyList()));
        }

        List<DNSRecord> records = lookupResult.getRight();
        for (DNSRecord record : records) {
            String recordString = record.getData();
            log.info("CNAME {} record: {}", lookupName, recordString);
            if (ALLOWED_CNAMES.contains(recordString)) {
                log.info("Verified host {} with record {}", hostId, recordString);
                return Optional.empty();
            }
        }

        log.info("No verification CNAME records for host {}", hostId);
        return Optional.of(new DNSRecordCNAMENotFoundFailInfo(true, records));
    }

    private Optional<VerificationFailInfo> tryVerifyHtmlFile(WebmasterHostId hostId, long verificationUin, VerificationCausedBy verificationCausedBy) {
        URL urlAddress = getVerificationUrl(hostId, verificationUin);

        log.info("Downloading {}", urlAddress);
        Either<ZoraUrlFetchResponse, VerificationFailInfo> downloadResult =
                verificationPageDownloadService.getVerificationPage(
                        urlAddress,
                        1,
                        EnumSet.of(
                                RedirectResolveFeature.ALLOW_RELATIVE,
                                RedirectResolveFeature.ALLOW_SWITCH_PORT,
                                RedirectResolveFeature.ALLOW_SWITCH_SCHEMA,
                                RedirectResolveFeature.ALLOW_SWITCH_WWW)
                );

        boolean hasFetchResponse = downloadResult.isLeft();
        if (!hasFetchResponse) {
            log.info("Failed to download {}", urlAddress);
            return downloadResult.getRight();
        }

        log.info("Downloaded {}", urlAddress);
        ZoraUrlFetchResponse zoraUrlFetchResponse = downloadResult.getLeftUnsafe();
        try {
            String fileContent = ZoraResponseDocumentUtil.getResponseString(zoraUrlFetchResponse);
            String secret = UinUtil.getUinString(verificationUin);
            if (fileContent.length() <= 1024 && fileContent.contains(secret)) {
                log.info("Verified host {} with {}", hostId, urlAddress);
                return Optional.empty();
            } else {
                log.info("Secret not found for host {} in {}", hostId, urlAddress);
                VerificationHttpResponseInfo responseInfo = VerificationHttpResponseInfo.createFromZoraResponse(zoraUrlFetchResponse);
                return Optional.of(new WrongHtmlPageContentFailInfo(responseInfo));
            }
        } catch (IOException | InternalException e) {
            log.error("Error processing {}", urlAddress);
            VerificationHttpResponseInfo responseInfo = VerificationHttpResponseInfo.createFromZoraResponse(zoraUrlFetchResponse);
            return Optional.of(
                    new PageUnavailableFailInfo(
                            urlAddress,
                            responseInfo,
                            PageUnavailableReason.CONTENT_PROCESSING_ERROR
                    )
            );
        }
    }

    private static URL getVerificationUrl(WebmasterHostId hostId, long verificationUin) {
        try {
            return new URL(IdUtils.hostIdToUrl(hostId) + "/" + UinUtil.getUinString(verificationUin) + ".html");
        } catch (MalformedURLException e) {
            throw new RuntimeException("Unexpected error: failed to build html file verification url for host " + hostId + " uin " + verificationUin);
        }
    }
}
