package ru.yandex.webmaster3.storage.verification;

import com.google.common.collect.Sets;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.GetEmailsType;
import ru.yandex.webmaster3.core.blackbox.UserEmailInfo;
import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.verification.IUserHostVerifier;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.NotApplicableFailInfo;
import ru.yandex.webmaster3.core.host.verification.fail.VerifiedEmailsNotFoundInWhoisFailInfo;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.verification.whois.ContactInfoWhoisHandler;
import ru.yandex.webmaster3.storage.verification.whois.SimpleWhoisHandler;
import ru.yandex.webmaster3.storage.verification.whois.WhoisRecordHandler;
import ru.yandex.webmaster3.storage.verification.whois.WhoisRecordReader;

import java.io.IOException;
import java.util.*;

/**
 * @author avhaliullin
 */
@Component
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class WhoisVerifier implements IUserHostVerifier {
    private static final Logger log = LoggerFactory.getLogger(WhoisVerifier.class);

    private static final Set<String> UNSUPPORTED_DOMAINS = Sets.newHashSet(
            "ru", "by", "es", "kz", "design", "az", "club", "life", IdUtils.IDN.toASCII("рф"), IdUtils.IDN.toASCII("москва")
    );
    private static final List<WhoisRecordHandler> RECORD_HANDLERS = Arrays.asList(
            new ContactInfoWhoisHandler(),
            new SimpleWhoisHandler()
    );

    private final BlackboxUsersService blackboxExternalYandexUsersService;

    @Override
    public Optional<VerificationFailInfo> verify(long userId, WebmasterHostId hostId, UUID recordId, long verificationUin, VerificationCausedBy verificationCausedBy) {
        String verifyForHost = verifyForHostName(hostId);
        if (verifyForHost == null) {
            return Optional.of(new NotApplicableFailInfo());
        }

        List<UserEmailInfo> allEmails = blackboxExternalYandexUsersService.getUserEmails(userId, GetEmailsType.ALL);
        Set<String> verifiedEmails = new HashSet<>();
        Set<String> addedNotVerifiedEmails = new HashSet<>();
        for (UserEmailInfo userEmailInfo : allEmails) {
            if (userEmailInfo.isValidatedEmail()) {
                verifiedEmails.add(userEmailInfo.getEmail().toLowerCase());
            } else {
                addedNotVerifiedEmails.add(userEmailInfo.getEmail());
            }
        }

        Set<String> foundEmails = new HashSet<>();
        try (WhoisRecordReader whoisRecordReader = new WhoisRecordReader(verifyForHost)) {
            String line;
            while ((line = whoisRecordReader.readLine()) != null) {
                for (WhoisRecordHandler handler : RECORD_HANDLERS) {
                    String extractedEmail = handler.nextLine(line);
                    if (extractedEmail != null) {
                        if (verifiedEmails.contains(extractedEmail.toLowerCase())) {
                            return Optional.empty();
                        } else {
                            foundEmails.add(extractedEmail);
                        }
                    }
                }
            }
        } catch (IOException e) {
            throw new WebmasterException("Failed to iterate whois records",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
        }
        return Optional.of(new VerifiedEmailsNotFoundInWhoisFailInfo(verifiedEmails, addedNotVerifiedEmails, foundEmails));
    }

    @Override
    public boolean isApplicable(WebmasterHostId hostId, long userId) {
        //return verifyForHostName(hostId) != null;
        return false; // WMC-10270 - прячем WHOIS от юзеров
    }

    private String verifyForHostName(WebmasterHostId hostId) {
        String hostName = hostId.getPunycodeHostname();
        String[] parts = hostName.split("\\.");
        if (parts.length > 3) {
            return null;
        }
        if (parts.length == 3) {
            if (parts[0].equals("www")) {
                parts = new String[]{parts[1], parts[2]};
                hostName = parts[0] + "." + parts[1];
            } else {
                return null;
            }
        }
        if (UNSUPPORTED_DOMAINS.contains(parts[1])) {
            return null;
        }
        return hostName;
    }
}
