package ru.yandex.webmaster3.storage.video;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.checklist.dao.RealTimeSiteProblemsYDao;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.data.RealTimeSiteProblemInfo;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;

/**
 * @author vsedaikina
 * 18.10.21
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class VideohostOfferService {

    private final VideohostOffersCurrentYDao videohostOffersCurrentYDao;
    private final VideohostOffersHistoryYDao videohostOffersHistoryYDao;
    private final WMCEventsService wmcEventsService;
    private final AbtService abtService;
    private final RealTimeSiteProblemsYDao realTimeSiteProblemsYDao;
    private final SiteProblemsService siteProblemsService;
    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;

    private final List<SiteProblemTypeEnum> allOfferAlerts = List.of(SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED,
                                                            SiteProblemTypeEnum.VIDEOHOST_OFFER_IS_NEEDED,
                                                            SiteProblemTypeEnum.VIDEOHOST_OFFER_NEED_PAPER);

    public VideohostOffer getVideohostOffer(WebmasterHostId hostId) {
        String hostUrl = WwwUtil.cutWWWAndM(hostId);
        log.info("Get videohost offer for host {}", hostUrl);
        VideohostOffer offer = videohostOffersCurrentYDao.getOffer(hostUrl);
        if (offer == null) {
            offer = VideohostOffer.builder().status(OfferStatus.NO_OFFER).build();
        }
        return offer;
    }

    public void removeVideohostOffer(WebmasterHostId hostId) {
        removeVideohostOffer(hostId, "Offer has been removed");
        //should send offer_is_needed, but maybe not right away
    }

    private void removeVideohostOffer(WebmasterHostId hostId, String comment) {
        String hostUrl = WwwUtil.cutWWWAndM(hostId);
        log.info("Remove videohost offer for host {}", hostUrl);
        dropAlert(hostId, SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED);
        VideohostOffer videohostOffer = videohostOffersCurrentYDao.removeOffer(hostUrl);

        if (videohostOffer != null) {
            videohostOffersHistoryYDao.insert(videohostOffer.toBuilder()
                    .status(OfferStatus.REMOVED)
                    .error(OfferErrorType.EMPTY)
                    .comments(comment)
                    .updateDate(DateTime.now())
                    .build()
            );
        }
    }

    public void updateVideohostOfferUrl(WebmasterHostId hostId, String newOfferUrl, boolean isRechecking) {
        //if new - create, if exists - remove and create
        String hostUrl = WwwUtil.cutWWWAndM(hostId);
        VideohostOffer existingOffer = videohostOffersCurrentYDao.getOffer(hostUrl);

        dropAlert(hostId, SiteProblemTypeEnum.VIDEOHOST_OFFER_IS_NEEDED, SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED);

        if (existingOffer != null && !isRechecking) {
            removeVideohostOffer(hostId, "Offer has been replaced");
        }

        log.info("Create new videohost offer for host {}", hostUrl);

        String comment = "New offer has been added";
        if (isRechecking) {
            comment = "Fixed offer for re-checking";
        }

        VideohostOffer newVideohostOffer = new VideohostOffer(
                hostUrl,
                newOfferUrl,
                OfferStatus.IN_PROGRESS,
                DateTime.now(),
                OfferErrorType.EMPTY,
                comment,
                DateTime.now()
        );

        videohostOffersCurrentYDao.insert(newVideohostOffer);
        videohostOffersHistoryYDao.insert(newVideohostOffer);
    }

    public void updateVideohostOffersStatus(List<VideohostOffer> offers) {
        for (VideohostOffer offer : offers) {
            updateVideohostOfferStatus(offer);
        }
    }

    private void updateVideohostOfferStatus(VideohostOffer offer) {
        log.info("Update status for offer {}", offer.getHostUrl());
        VideohostOffer oldOffer = videohostOffersCurrentYDao.getOffer(offer.getHostUrl());
        DateTime offerAddDate = checkIncomingDate(offer.getAddDate());
        if (oldOffer != null) {
            if (offerAddDate.isBefore(oldOffer.getAddDate())) {
                log.info("Skip outdated offers");
                return;
            }

            if (offer.getAddDate().isEqual(oldOffer.getAddDate()) || offerAddDate.isEqual(oldOffer.getAddDate())) {
                if (offer.getStatus() == oldOffer.getStatus()) {
                    log.info("Skip offers which status didn't change");
                    return;
                }
            }
        }

        sendMessage(offer);

        if (oldOffer != null) {
            log.info("Remove offer with old status");
//            videohostOffersCurrentYDao.removeOffer(offer.getHostUrl()); //might be redundant
        }
        VideohostOffer newOffer = offer.toBuilder().addDate(offerAddDate).updateDate(DateTime.now()).build();

        log.info("Update videohost offer status for host {}, new status {}", newOffer.getHostUrl(), newOffer.getStatus());

        videohostOffersCurrentYDao.insert(newOffer);
        videohostOffersHistoryYDao.insert(newOffer);
    }

    private void sendMessage(VideohostOffer offer) {
        WebmasterHostId hostId = IdUtils.urlToHostId(offer.getHostUrl());

        switch (offer.getStatus()) {
            case FAILED -> {
                log.info("Send videohost_offer_is_failed to {}", hostId);
                dropAlert(hostId, SiteProblemTypeEnum.VIDEOHOST_OFFER_IS_NEEDED, SiteProblemTypeEnum.VIDEOHOST_OFFER_NEED_PAPER);

                SiteProblemContent content = new SiteProblemContent.VideohostOfferFailed(offer.getOfferUrl());
                DateTime lastUpdate = DateTime.now();
                siteProblemsService.updateRealTimeProblem(hostId, new ProblemSignal(content, lastUpdate));

            }
            case CONFIRMED -> {
                log.info("Send videohost_offer_confirmed to {}", hostId);
                dropAlert(hostId, allOfferAlerts);

                wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                        hostId,
                        new MessageContent.VideohostOfferConfirmed(
                                hostId,
                                offer.getOfferUrl()
                        ),
                        NotificationType.VIDEOHOST_OFFER,
                        false)
                )));
            }
            case IN_PROGRESS -> {
                dropAlert(hostId, allOfferAlerts);
            }
            case NEED_PAPER -> {
                log.info("Send videohost_offer_need_paper to {}", hostId);
                dropAlert(hostId, SiteProblemTypeEnum.VIDEOHOST_OFFER_IS_NEEDED, SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED);

                SiteProblemContent content = new SiteProblemContent.VideohostOfferNeedPaper();
                DateTime lastUpdate = DateTime.now();
                siteProblemsService.updateRealTimeProblem(hostId, new ProblemSignal(content, lastUpdate));
            }
            case HAS_PAPER -> {
                log.info("Send videohost_offer_has_paper to {}", hostId);
                dropAlert(hostId, allOfferAlerts);

                wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                        hostId,
                        new MessageContent.VideohostOfferHasPaper(hostId),
                        NotificationType.VIDEOHOST_OFFER,
                        false)
                )));
            }
            case REDUNDANT -> {
                log.info("Send videohost_offer_redundant to {}", hostId);
                dropAlert(hostId, allOfferAlerts);

                wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                        hostId,
                        new MessageContent.VideohostOfferRedundant(hostId),
                        NotificationType.VIDEOHOST_OFFER,
                        false)
                )));
            }
            default -> {
                //If an offer from outside has wrong status
                log.info("Unexpected offer status type {} for host {}", offer.getStatus(), offer.getHostUrl());
            }
        }
    }

    private DateTime checkIncomingDate(DateTime date) {
        //sometimes might be in sec instead of millis
        long millis = date.getMillis();
        millis = (millis < Math.pow(10, 10)) ? millis * 1000 : millis;
        return new DateTime(millis);
    }

    public void sendVideohostOfferIsNeeded(Collection<WebmasterHostId> hosts) {
        removeHostsWithAlerts(hosts);
        hosts.forEach(this::sendVideohostOfferIsNeeded);
    }

    private void removeHostsWithAlerts(Collection<WebmasterHostId> alertedHosts) {
        Map<WebmasterHostId, RealTimeSiteProblemInfo> map;

        for (SiteProblemTypeEnum siteProblem : allOfferAlerts) {
            map = siteProblemsService.listSitesProblems(alertedHosts, siteProblem);
            alertedHosts.removeAll(map.keySet());
        }
    }

    private void sendVideohostOfferIsNeeded(WebmasterHostId hostId) {
        if (!abtService.isInExperiment(hostId, "VIDEO_HOST_OFFER")) { //actually redundant after modifications
            log.info("Host {} is not in experiment", hostId);
            return;
        }

        OfferStatus status = videohostOffersCurrentYDao.getOfferStatus(WwwUtil.cutWWWAndM(hostId));
        if (status != null) { //failed or confirmed + needPaper 'n hasPaper now + redundant
            log.info("Host {} already has status {}", hostId, status);
            //not sure if alert is needed in this case
            return;
        }

        log.info("Send videohost_offer_is_needed to {}", hostId);
        dropAlert(hostId, SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED, SiteProblemTypeEnum.VIDEOHOST_OFFER_NEED_PAPER);

        SiteProblemContent content = new SiteProblemContent.VideohostOfferIsNeeded();
        DateTime lastUpdate = DateTime.now();
        siteProblemsService.updateRealTimeProblem(hostId, new ProblemSignal(content, lastUpdate));
    }

    public void dropAlert(WebmasterHostId hostId, SiteProblemTypeEnum... problemType) {
        dropAlert(hostId, Arrays.asList(problemType));
    }

    private void dropAlert(WebmasterHostId hostId, List<SiteProblemTypeEnum> problemType) {
        realTimeSiteProblemsYDao.deleteProblems(IdUtils.urlToHostId(WwwUtil.cutWWWAndM(hostId)), problemType);
    }

    public boolean isDomainInWebmaster(String hostUrl) {
        List<WebmasterHostId> problemHosts = (IdUtils.allHostsForDomainWithM(IdUtils.urlToHostId(WwwUtil.cutWWWAndM(hostUrl))));
        return problemHosts.stream().anyMatch(allVerifiedHostsCacheService::contains);
    }
}
