package ru.yandex.webmaster3.storage.video;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.stereotype.Repository;
import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.ValueDataMapper;

import java.util.List;
import java.util.function.Consumer;

/**
 * @author vsedaikina
 * 18.10.21
 */
@Repository
public class VideohostOffersCurrentYDao extends AbstractYDao {

    protected VideohostOffersCurrentYDao() {
        super(PREFIX_VIDEO, "videohost_offers_current");
    }

    public void insert(VideohostOffer vho) {
        upsert(
                F.HOST_URL.value(vho.getHostUrl()),
                F.OFFER_URL.value(vho.getOfferUrl()),
                F.STATUS.value(vho.getStatus()),
                F.ADD_DATE.value(vho.getAddDate()),
                F.ERROR.value(vho.getError()),
                F.COMMENTS.value(vho.getComments())
        ).execute();
    }

    public OfferStatus getOfferStatus(String hostUrl) {
        return select(F.STATUS)
                .where(F.HOST_URL.eq(hostUrl))
                .queryOne();
    }

    public VideohostOffer removeOffer(String hostUrl) {
        VideohostOffer offer = getOffer(hostUrl);
        if (offer == null) {
            return null;
        }

        delete()
                .where(F.HOST_URL.eq(hostUrl))
                .execute();

        return offer;
    }

    public VideohostOffer getOffer(String hostUrl) {
        return select(MAPPER)
                .where(F.HOST_URL.eq(hostUrl))
                .queryOne();
    }

    public void forEach(Consumer<VideohostOffer> consumer) {
        streamReader(MAPPER, consumer);
    }

    private static final DataMapper<VideohostOffer> MAPPER = DataMapper.create(
            F.HOST_URL,
            F.OFFER_URL,
            F.STATUS,
            F.ADD_DATE,
            F.ERROR,
            F.COMMENTS,
            (hostUrl, offerUrl, status, addDate, error, comments) -> VideohostOffer.builder().hostUrl(hostUrl)
                    .offerUrl(offerUrl).status(status).addDate(addDate).error(error).comments(comments).build()
    );

    public void batchDelete(List<VideohostOffer> items) {
        batchDelete(VALUE_MAPPER, items).execute();
    }

    public void batchInsert(List<VideohostOffer> items) {
        batchInsert(VALUE_MAPPER, items).execute();
    }

    private static final ValueDataMapper<VideohostOffer> VALUE_MAPPER = ValueDataMapper.create2(
            Pair.of(F.HOST_URL, VideohostOffer::getHostUrl),
            Pair.of(F.OFFER_URL, VideohostOffer::getOfferUrl),
            Pair.of(F.STATUS, VideohostOffer::getStatus),
            Pair.of(F.ADD_DATE, VideohostOffer::getAddDate),
            Pair.of(F.ERROR, VideohostOffer::getError),
            Pair.of(F.COMMENTS, VideohostOffer::getComments)
    );

    private interface F {
        Field<String> HOST_URL = Fields.stringField("host_url");
        Field<String> OFFER_URL = Fields.stringField("offer_url").makeOptional();
        Field<OfferStatus> STATUS = Fields.stringEnumField("status", OfferStatus.R);
        Field<DateTime> ADD_DATE = Fields.jodaDateTimeField("add_date");
        Field<OfferErrorType> ERROR = Fields.stringEnumField("error", OfferErrorType.R).makeOptional();
        Field<String> COMMENTS = Fields.stringField("comment").makeOptional();
    }
}
