package ru.yandex.webmaster3.storage.ytimport;

import java.util.Collections;
import java.util.List;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.core.type.TypeReference;
import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHostLocation;
import ru.yandex.webmaster3.storage.util.yt.YtPath;

/**
 * @author avhaliullin
 */
public class YtClickhouseImportState {
    private final UUID taskId;
    private final List<YtClickhouseTableRelation> tables;
    private final List<Subtask> subtasks;
    private final String database;
    private final String tableInsertSpec;
    private final YtClickhouseImportStateEnum state;
    private final Instant lastUpdate;
    private final ImportPriority priority;

    public YtClickhouseImportState(UUID taskId, List<YtClickhouseTableRelation> tables, List<Subtask> subtasks,
                                   String database, String tableInsertSpec, YtClickhouseImportStateEnum state,
                                   Instant lastUpdate, ImportPriority priority) {
        this.taskId = taskId;
        this.tables = tables;
        this.subtasks = subtasks;
        this.database = database;
        this.tableInsertSpec = tableInsertSpec;
        this.state = state;
        this.lastUpdate = lastUpdate;
        this.priority = priority;
    }

    public List<YtClickhouseTableRelation> getTables() {
        return tables;
    }

    public String getTableInsertSpec() {
        return tableInsertSpec;
    }

    public UUID getTaskId() {
        return taskId;
    }

    public DateTime getCreateDate() {
        return new DateTime(UUIDs.unixTimestamp(taskId));
    }

    public List<Subtask> getSubtasks() {
        return subtasks;
    }

    public String getDatabase() {
        return database;
    }

    public YtClickhouseImportStateEnum getState() {
        return state;
    }

    public Instant getLastUpdate() {
        return lastUpdate;
    }

    public ImportPriority getPriority() {
        return priority;
    }

    public YtClickhouseImportState started(List<Subtask> subtasks) {
        return new YtClickhouseImportState(
                taskId, tables, subtasks, database, tableInsertSpec, YtClickhouseImportStateEnum.STARTED,
                Instant.now(), priority);
    }

    public YtClickhouseImportState fatalError() {
        return new YtClickhouseImportState(
                taskId, tables, subtasks, database, tableInsertSpec, YtClickhouseImportStateEnum.FAILED,
                Instant.now(), priority);
    }

    public YtClickhouseImportState withSubtasks(List<Subtask> keepTrackingSubtasks) {
        return new YtClickhouseImportState(
                taskId, tables, keepTrackingSubtasks, database, tableInsertSpec, state, Instant.now(),
                priority);
    }

    public YtClickhouseImportState finished() {
        return new YtClickhouseImportState(
                taskId, tables, Collections.emptyList(), database, tableInsertSpec, YtClickhouseImportStateEnum.DONE, Instant.now(),
                priority);
    }

    public YtClickhouseImportState cancelled() {
        return new YtClickhouseImportState(
                taskId, tables, subtasks, database, tableInsertSpec, YtClickhouseImportStateEnum.CANCELLED,
                Instant.now(), priority);
    }

    public static class Subtask {
        public static final TypeReference<List<Subtask>> LIST_REFERENCE = new TypeReference<>() {
        };
        private final UUID subtaskId;
        private final YtPath ytPath;
        private final ClickhouseHostLocation host;
        private final String tableName;
        private final String createSpec;

        @JsonCreator
        public Subtask(UUID subtaskId, YtPath ytPath, ClickhouseHostLocation host, String tableName, String createSpec) {
            this.subtaskId = subtaskId;
            this.ytPath = ytPath;
            this.host = host;
            this.tableName = tableName;
            this.createSpec = createSpec;
        }

        public UUID getSubtaskId() {
            return subtaskId;
        }

        public YtPath getYtPath() {
            return ytPath;
        }

        public ClickhouseHostLocation getHost() {
            return host;
        }

        public String getTableName() {
            return tableName;
        }

        public String getCreateSpec() {
            return createSpec;
        }

        @Override
        public String toString() {
            return "{" +
                    "subtaskId=" + subtaskId +
                    ", ytPath=" + ytPath +
                    ", host=" + host +
                    ", tableName='" + tableName + '\'' +
                    '}';
        }
    }
}
