package ru.yandex.webmaster3.storage.ytimport.dao;

import java.util.Comparator;
import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Value;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.Instant;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.storage.util.ydb.AbstractYDao;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.DataMapper;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Field;
import ru.yandex.webmaster3.storage.util.ydb.querybuilder.typesafe.Fields;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.ytimport.ImportPriority;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportStateEnum;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseTableRelation;

/**
 * @author avhaliullin
 */
@Repository
public class YtClickhouseImportQueueYDao extends AbstractYDao {
    //private static final Duration TTL = Duration.standardDays(7);

    private static final DataMapper<YtClickhouseImportRecord> MAPPER = DataMapper.create(
            F.ID, F.DATABASE, F.TABLES, F.CH_INSERT_SPEC, F.STATE, F.LAST_UPDATE, F.PRIORITY,
            YtClickhouseImportRecord::new
    );

    public YtClickhouseImportQueueYDao() {
        super(PREFIX_IMPORTER, "yt_clickhouse_import_queue");
    }

    public void update(YtClickhouseImportRecord state) {
        upsert(
                F.ID.value(state.getId()),
                F.DATABASE.value(state.getDatabase()),
                F.LAST_UPDATE.value(state.getLastUpdate()),
                F.CH_INSERT_SPEC.value(state.getChInsertSpec()),
                F.STATE.value(state.getState()),
                F.TABLES.value(state.getTables()),
                F.PRIORITY.value(state.getPriority())
        ).execute();
    }

    public List<YtClickhouseImportRecord> listAll() {
        return select(MAPPER).queryForList(
                Pair.of(F.ID, r -> r.id)
        );
    }

    public List<YtClickhouseImportRecord> listActive() {
        var lst = select(MAPPER)
                .where(F.STATE.in(YtClickhouseImportStateEnum.NEW, YtClickhouseImportStateEnum.STARTED))
                .queryForList(
                    Pair.of(F.ID, r -> r.id)
                );

        lst.sort(Comparator.comparing(YtClickhouseImportRecord::getPriority));
        return lst;
    }

    public YtClickhouseImportRecord getTask(UUID taskId) {
        return select(MAPPER).where(F.ID.eq(taskId)).queryOne();
    }

    public void deleteTask(UUID taskId) {
        delete().where(F.ID.eq(taskId)).execute();
    }

    @Getter
    @AllArgsConstructor
    @Builder(toBuilder = true)
    @EqualsAndHashCode(of = {"ytTable", "chTable", "shard", "dc"})
    public static class YtClickhouseImportTable {
        public static final TypeReference<List<YtClickhouseImportTable>> LIST_REFERENCE = new TypeReference<>() {
        };
        YtPath ytTable;
        String chTable;
        String chCreateSpec;
        Integer shard;
        String dc;
        boolean done;

        public static YtClickhouseImportTable fromTableRelation(YtClickhouseTableRelation tr) {
            return fromTableRelation(tr, null);
        }

        public static YtClickhouseImportTable fromTableRelation(YtClickhouseTableRelation tr, String dc) {
            return new YtClickhouseImportTable(tr.getYtTable(), tr.getClickhouseTableName(), tr.getClickhouseTableCreateSpec(),
                    tr.getClickhouseShard(), dc, false);
        }
    }

    @Value
    @Builder(toBuilder = true)
    public static class YtClickhouseImportRecord {
        UUID id;
        String database;
        List<YtClickhouseImportTable> tables;
        String chInsertSpec;
        YtClickhouseImportStateEnum state;
        Instant lastUpdate;
        ImportPriority priority;

        public YtClickhouseImportState toState() {
            return new YtClickhouseImportState(id, null, null, database, chInsertSpec, state, lastUpdate, priority);
        }
    }

    private interface F {
        Field<UUID> ID = Fields.uuidField("id");
        Field<String> DATABASE = Fields.stringField("database");
        Field<Instant> LAST_UPDATE = Fields.jodaInstantField("last_update");
        Field<String> CH_INSERT_SPEC = Fields.stringField("ch_insert_spec");
        Field<YtClickhouseImportStateEnum> STATE = Fields.stringEnumField("state", YtClickhouseImportStateEnum.R);
        Field<List<YtClickhouseImportTable>> TABLES = Fields.jsonField2("tables", YtClickhouseImportTable.LIST_REFERENCE);
        Field<ImportPriority> PRIORITY = Fields.intEnumField("priority", ImportPriority.R);
    }
}
