package ru.yandex.webmaster3.streamer.feeds.mbi;

import java.util.Collections;

import com.google.common.base.Strings;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.market.core.logbroker.FeedParsingResultOuterClass.FeedParsingResult;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedStatus;
import ru.yandex.webmaster3.core.logbroker.reader.IDataProcessing;
import ru.yandex.webmaster3.core.logbroker.reader.MessageContainer;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;

/**
 * Created by Oleg Bazdyrev on 01/09/2021.
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class FeedParsingResultsProcessing implements IDataProcessing {

    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsService feedsService;

    @Override
    public void process(MessageContainer messageContainer) {
        for (byte[] rawMessage : messageContainer.getRawMessages()) {
            try {
                FeedParsingResult result = FeedParsingResult.parseFrom(rawMessage);
                log.info("Got feed parsing result for businessId = {} and partnerId = {}", result.getBusinessId(),
                        result.getPartnerId());
                NativeFeedInfo2 feedInfo = feedsService.findById(result.getBusinessId(), result.getPartnerId());
                if (feedInfo == null) {
                    log.warn("Feed info for businessId = {} and partnerId = {} is not found", result.getBusinessId(),
                            result.getPartnerId());
                } else if (feedInfo.getBusinessId() == null) {
                    log.warn("Feed info for businessId = {} and partnerId = {} is not mbi", result.getBusinessId(),
                            result.getPartnerId());
                } else {
                    var builder = feedInfo.toBuilder().statusVertical(convertStatus(result.getStatus()));
                    if (!Strings.isNullOrEmpty(result.getError())) {
                        builder.errors(Collections.singletonList(result.getError()));
                    } else {
                        builder.errors(Collections.emptyList());
                    }
                    // update
                    feedsNative2YDao.update(builder.build());
                }
            } catch (InvalidProtocolBufferException e) {
                log.error("Bad protobuf message", e);
            }
        }
        messageContainer.commit();
    }

    private static NativeFeedStatus convertStatus(FeedParsingResult.Status status) {
        return switch (status) {
            case NEW -> NativeFeedStatus.IN_PROGRESS;
            case OK -> NativeFeedStatus.SUCCESS;
            case ERROR -> NativeFeedStatus.FAILED;
            default -> throw new IllegalStateException();
        };
    }
}
