package ru.yandex.webmaster3.tanker.digest.html;

import org.w3c.dom.Document;
import org.w3c.dom.DocumentFragment;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author avhaliullin
 */
public abstract class HtmlNode {

    abstract Node toDom(Document document);

    public static class TextNode extends HtmlNode {
        private final String text;
        private final boolean unescape;

        public TextNode(String text, boolean unescape) {
            this.text = text;
            this.unescape = unescape;
        }

        @Override
        Node toDom(Document document) {
            if (unescape) {
                return HtmlDomUtil.createUnescapedTextNode(document, text);
            } else {
                return document.createTextNode(text);
            }
        }
    }

    public static class ListNode extends HtmlNode {
        private final List<HtmlNode> items = new ArrayList<>();

        public ListNode addItem(HtmlNode item) {
            items.add(item);
            return this;
        }

        public ListNode addItems(List<HtmlNode> items) {
            this.items.addAll(items);
            return this;
        }

        public ListNode addItems(HtmlNode... items) {
            return addItems(Arrays.asList(items));
        }

        @Override
        Node toDom(Document document) {
            DocumentFragment fragment = document.createDocumentFragment();
            items.forEach(item -> fragment.appendChild(item.toDom(document)));
            return fragment;
        }
    }

    public static abstract class TagNode<T extends TagNode<?>> extends HtmlNode {
        private final String tagName;
        private CSS style = new CSS();
        private HtmlSize width;
        private HtmlSize height;

        protected TagNode(String tagName) {
            this.tagName = tagName;
        }

        protected abstract T getSelf();

        public T addStyle(String name, String value) {
            this.style.set(name, value);
            return getSelf();
        }

        public T addStyle(CSS style) {
            this.style.add(style);
            return getSelf();
        }

        public T styleWidth(HtmlSize width) {
            this.style.width(width);
            return getSelf();
        }

        public T width(HtmlSize width) {
            this.width = width;
            return getSelf();
        }

        public T height(HtmlSize height) {
            this.height = height;
            return getSelf();
        }

        @Override
        Element toDom(Document document) {
            Element res = document.createElement(tagName);
            if (style != null) {
                String styleSerialized = style.serialize();
                if (!styleSerialized.isEmpty()) {
                    res.setAttribute("style", styleSerialized);
                }
            }
            if (width != null) {
                res.setAttribute("width", width.serialize());
            }
            if (height != null) {
                res.setAttribute("height", height.serialize());
            }
            return res;
        }
    }

    public static abstract class ContentNode<T extends ContentNode<?>> extends TagNode<T> {
        private final List<HtmlNode> content = new ArrayList<>();

        public ContentNode(String tagName) {
            super(tagName);
        }

        public T addContent(HtmlNode... content) {
            return addContent(Arrays.asList(content));
        }

        public T addContent(List<HtmlNode> content) {
            this.content.addAll(content);
            return getSelf();
        }

        @Override
        Element toDom(Document document) {
            Element result = super.toDom(document);
            content.forEach(node -> result.appendChild(node.toDom(document)));
            return result;
        }
    }

    // Trivial

    public static class BTag extends ContentNode<BTag> {
        public BTag() {
            super("b");
        }

        @Override
        protected BTag getSelf() {
            return this;
        }
    }

    public static class BrTag extends ContentNode<BrTag> {
        public BrTag() {
            super("br");
        }

        @Override
        protected BrTag getSelf() {
            return this;
        }
    }

    public static class HrTag extends TagNode<HrTag> {
        public HrTag() {
            super("hr");
        }

        @Override
        protected HrTag getSelf() {
            return this;
        }
    }

    public static class DivTag extends ContentNode<DivTag> {
        public DivTag() {
            super("div");
        }

        @Override
        protected DivTag getSelf() {
            return this;
        }
    }

    public static class SpanTag extends ContentNode<SpanTag> {
        public SpanTag() {
            super("span");
        }

        @Override
        protected SpanTag getSelf() {
            return this;
        }
    }

    public static class HTag extends ContentNode<HTag> {
        public HTag(int level) {
            super("h" + level);
        }

        @Override
        protected HTag getSelf() {
            return this;
        }
    }

    public static class LiTag extends ContentNode<LiTag> {
        public LiTag() {
            super("li");
        }

        @Override
        protected LiTag getSelf() {
            return this;
        }
    }

    public static class PTag extends ContentNode<PTag> {
        public PTag() {
            super("p");
        }

        @Override
        protected PTag getSelf() {
            return this;
        }
    }

    // Non-trivial

    public static class ATag extends ContentNode<ATag> {
        private String href;

        public ATag() {
            super("a");
        }

        @Override
        protected ATag getSelf() {
            return this;
        }

        public ATag href(String href) {
            this.href = href;
            return this;
        }

        @Override
        Element toDom(Document document) {
            Element result = super.toDom(document);
            if (href != null) {
                result.setAttribute("href", href);
            }
            return result;
        }
    }

    public static class ImgTag extends TagNode<ImgTag> {
        private String src;
        private String alt;

        public ImgTag(String src) {
            super("img");
            this.src = src;
        }

        public ImgTag src(String src) {
            this.src = src;
            return this;
        }

        public ImgTag alt(String alt) {
            this.alt = alt;
            return this;
        }

        @Override
        protected ImgTag getSelf() {
            return this;
        }

        @Override
        Element toDom(Document document) {
            Element result = super.toDom(document);
            if (src != null) {
                result.setAttribute("src", src);
            }
            if (alt != null) {
                result.setAttribute("alt", alt);
            }
            return result;
        }
    }

    public static class UlTag extends TagNode<UlTag> {
        private final List<LiTag> items = new ArrayList<>();

        public UlTag() {
            super("ul");
        }

        public UlTag addItems(List<LiTag> item) {
            items.addAll(item);
            return this;
        }

        public UlTag addItems(LiTag... item) {
            return addItems(Arrays.asList(item));
        }

        @Override
        protected UlTag getSelf() {
            return this;
        }

        @Override
        Element toDom(Document document) {
            Element ul = super.toDom(document);
            items.forEach(item -> ul.appendChild(item.toDom(document)));
            return ul;
        }
    }

    // Tables

    public static abstract class TagCellNode<T extends TagCellNode<T>> extends ContentNode<T> {
        public TagCellNode(String tagName) {
            super(tagName);
        }
    }

    public static class TdTag extends TagCellNode<TdTag> {
        public TdTag() {
            super("td");
        }

        @Override
        protected TdTag getSelf() {
            return this;
        }
    }

    public static class ThTag extends TagCellNode<ThTag> {
        public ThTag() {
            super("th");
        }

        @Override
        protected ThTag getSelf() {
            return this;
        }
    }

    public static abstract class TableRowNode<R extends TagCellNode<R>, T extends TableRowNode<?, ?>> extends TagNode<T> {
        private List<R> cells = new ArrayList<>();

        protected TableRowNode() {
            super("tr");
        }

        public R addCell() {
            R cell = createCell();
            addCell(cell);
            return cell;
        }

        public T addCell(R cell) {
            cells.add(cell);
            return getSelf();
        }

        public T addCells(List<R> cells) {
            this.cells.addAll(cells);
            return getSelf();
        }

        protected abstract R createCell();

        @Override
        Element toDom(Document document) {
            Element tr = super.toDom(document);
            cells.forEach(cell -> tr.appendChild(cell.toDom(document)));
            return tr;
        }
    }

    public static class TrDTag extends TableRowNode<TdTag, TrDTag> {
        @Override
        protected TrDTag getSelf() {
            return this;
        }

        @Override
        protected TdTag createCell() {
            return new TdTag();
        }
    }

    public static class TrHTag extends TableRowNode<ThTag, TrHTag> {
        @Override
        protected TrHTag getSelf() {
            return this;
        }

        @Override
        protected ThTag createCell() {
            return new ThTag();
        }
    }

    public static class TableTag extends TagNode<TableTag> {
        private HtmlSize border;
        private HtmlSize cellpadding;
        private HtmlSize cellspacing;
        private List<TrHTag> header;
        private List<TrDTag> body;

        public TableTag() {
            super("table");
        }

        public TableTag border(HtmlSize border) {
            this.border = border;
            return this;
        }

        public TableTag cellpadding(HtmlSize cellpadding) {
            this.cellpadding = cellpadding;
            return this;
        }

        public TableTag cellspacing(HtmlSize cellspacing) {
            this.cellspacing = cellspacing;
            return this;
        }

        public TrHTag addHeader() {
            TrHTag row = new TrHTag();
            addHeader(row);
            return row;
        }

        public TableTag addHeader(ThTag... row) {
            return addHeader(Arrays.asList(row));
        }

        public TableTag addHeader(List<ThTag> row) {
            addHeader().addCells(row);
            return this;
        }

        public TableTag addHeader(TrHTag row) {
            if (header == null) {
                header = new ArrayList<>();
            }
            header.add(row);
            return this;
        }

        public TrDTag addRow() {
            TrDTag row = new TrDTag();
            addRows(row);
            return row;
        }

        public TableTag addRow(TdTag... row) {
            return addRow(Arrays.asList(row));
        }

        public TableTag addRow(List<TdTag> row) {
            addRow().addCells(row);
            return this;
        }

        public TableTag addRows(TrDTag... row) {
            return addRows(Arrays.asList(row));
        }

        public TableTag addRows(List<TrDTag> row) {
            if (body == null) {
                body = new ArrayList<>();
            }
            body.addAll(row);
            return this;
        }

        @Override
        protected TableTag getSelf() {
            return this;
        }

        @Override
        Element toDom(Document document) {
            Element result = super.toDom(document);
            if (border != null) {
                result.setAttribute("border", border.serialize());
            }
            if (cellspacing != null) {
                result.setAttribute("cellspacing", cellspacing.serialize());
            }
            if (cellpadding != null) {
                result.setAttribute("cellpadding", cellpadding.serialize());
            }
            if (header != null) {
                Element thead = document.createElement("thead");
                result.appendChild(thead);
                for (HtmlNode row : header) {
                    thead.appendChild(row.toDom(document));
                }
            }
            if (body != null) {
                Element tbody = document.createElement("tbody");
                result.appendChild(tbody);
                for (HtmlNode row : body) {
                    tbody.appendChild(row.toDom(document));
                }
            }
            return result;
        }
    }

    // static factories
    public static HtmlNode list(HtmlNode... items) {
        return new ListNode().addItems(items);
    }

    public static HtmlNode list(List<HtmlNode> items) {
        return new ListNode().addItems(items);
    }

    public static TrDTag trd(List<TdTag> cells) {
        return new TrDTag().addCells(cells);
    }
    public static TrDTag trd(TdTag... cells) {
        return trd(Arrays.asList(cells));
    }

    public static ATag a(String href) {
        return new ATag().href(href);
    }

    public static BrTag br(HtmlNode... content) {
        return new BrTag().addContent(content);
    }

    public static PTag p(HtmlNode... content) {
        return new PTag().addContent(content);
    }

    public static DivTag div() {
        return new DivTag();
    }

    public static SpanTag span() {
        return new SpanTag();
    }

    public static TdTag td(HtmlNode... content) {
        TdTag result = new TdTag();
        result.addContent(content);
        return result;
    }

    public static ThTag th(HtmlNode... content) {
        ThTag result = new ThTag();
        result.addContent(content);
        return result;
    }

    public static TextNode safeText(String text) {
        return new TextNode(text, true);
    }

    public static TextNode text(String text) {
        return new TextNode(text, false);
    }

    public static HTag h1(HtmlNode... content) {
        return new HTag(1)
                .addContent(content);
    }

    public static HTag h2(HtmlNode... content) {
        return new HTag(2)
                .addContent(content);
    }

    public static HTag h3(HtmlNode... content) {
        return new HTag(3)
                .addContent(content);
    }

    public static HTag header(int level, HtmlNode... content) {
        HTag result = new HTag(level);
        result.addContent(content);
        return result;
    }

    public static UlTag ul(List<LiTag> items) {
        return new UlTag().addItems(items);
    }

    public static HtmlNode EMPTY = new HtmlNode() {
        @Override
        Node toDom(Document document) {
            return document.createDocumentFragment();
        }
    };
}
