package ru.yandex.webmaster3.tanker.internal;

import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.text.templates.RenderableText;
import ru.yandex.webmaster3.core.text.templates.TextTemplate;
import ru.yandex.webmaster3.tanker.TankerDynamicKeyWithBuilder;
import ru.yandex.webmaster3.tanker.TankerStaticKey;
import ru.yandex.webmaster3.tanker.TankerStaticKeyWithBuilder;
import ru.yandex.webmaster3.tanker.digest.html.HtmlNode;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * @author avhaliullin
 */
public final class TankerKeyUtil {
    private TankerKeyUtil() {
    }

    public static DynamicBuilder newDynamicBuilder() {
        return new DynamicBuilder();
    }

    public static class DynamicBuilder {
        private final Map<LanguageEnum, LangBuilder> templates = new EnumMap<>(LanguageEnum.class);

        public LangBuilder language(LanguageEnum language) {
            return templates.computeIfAbsent(language, x -> new LangBuilder());
        }

        public <Builder> TankerDynamicKeyWithBuilder<Builder> build(Function<List<TankerTemplateEntry>, Builder> builderFactory) {
            Map<LanguageEnum, List<TankerTemplateEntry>> langs = new EnumMap<>(LanguageEnum.class);
            for (Map.Entry<LanguageEnum, LangBuilder> entry : templates.entrySet()) {
                langs.put(entry.getKey(), entry.getValue().doBuild());
            }
            return new TankerDynamicKeyWithBuilder<>(langs, builderFactory);
        }

        public class LangBuilder {
            private final List<TankerTemplateEntry> parts = new ArrayList<>();
            private int textLength = 0;
            private Map<String, Integer> phsCount = new HashMap<>();

            public LangBuilder addEntry(TankerTemplateEntry entry) {
                parts.add(entry);
                return this;
            }

            public LangBuilder addText(String text) {
                parts.add(new TankerTemplateEntry.TextEntry(text));
                return this;
            }

            public LangBuilder addPlaceholder(String ph) {
                parts.add(new TankerTemplateEntry.PlaceholderEntry(ph));
                return this;
            }

            private List<TankerTemplateEntry> doBuild() {
                return parts;
            }

            public <Builder> TankerDynamicKeyWithBuilder<Builder> build(Function<List<TankerTemplateEntry>, Builder> builderFactory) {
                return DynamicBuilder.this.build(builderFactory);
            }

            public LangBuilder language(LanguageEnum lang) {
                return DynamicBuilder.this.language(lang);
            }
        }
    }

    public static StaticBuilder newStaticBuilder() {
        return new StaticBuilder();
    }

    public static HtmlNode createHtmlNode(List<TankerTemplateEntry> template, TankerPlaceholders<HtmlNode> phs) {
        HtmlNode.ListNode listNode = new HtmlNode.ListNode();
        template.forEach(e -> addHtmlEntry(listNode, phs, e));
        return listNode;
    }

    public static void addHtmlEntry(HtmlNode.ListNode listNode, TankerPlaceholders<HtmlNode> phs, TankerTemplateEntry entry) {
        if (entry instanceof TankerTemplateEntry.TextEntry) {
            listNode.addItem(HtmlNode.safeText(((TankerTemplateEntry.TextEntry) entry).getText()));
        } else if (entry instanceof TankerTemplateEntry.PlaceholderEntry) {
            HtmlNode phValue = phs.getOrDefault(((TankerTemplateEntry.PlaceholderEntry) entry).getPlaceholder(), HtmlNode.EMPTY);
            listNode.addItem(phValue);
        } else if (entry instanceof TankerTemplateEntry.ConditionalEntry) {
            TankerTemplateEntry.ConditionalEntry conditionalEntry = (TankerTemplateEntry.ConditionalEntry) entry;
            processCondition(conditionalEntry, phs, e -> addHtmlEntry(listNode, phs, e));
        } else if (entry instanceof TankerTemplateEntry.PluralEntry) {
            TankerTemplateEntry.PluralEntry pluralEntry = (TankerTemplateEntry.PluralEntry) entry;
            processPlural(pluralEntry, phs, e -> addHtmlEntry(listNode, phs, e));
        }
    }

    public static RenderableText createRenderableText(List<TankerTemplateEntry> template, TankerPlaceholders<String> phs) {
        List<TextTemplate.Entry<String>> textEntries = new ArrayList<>();
        template.forEach(e -> addEntry(textEntries, phs, e));
        return new TextTemplate<>(textEntries, 0, Collections.emptyMap()).createRenderable(phs.getPhs());
    }

    private static void addEntry(List<TextTemplate.Entry<String>> textEntries, TankerPlaceholders<String> phs, TankerTemplateEntry entry) {
        if (entry instanceof TankerTemplateEntry.TextEntry) {
            textEntries.add(new TextTemplate.TextEntry<>(((TankerTemplateEntry.TextEntry) entry).getText()));
        } else if (entry instanceof TankerTemplateEntry.PlaceholderEntry) {
            textEntries.add(new TextTemplate.PlaceholderEntry<>(((TankerTemplateEntry.PlaceholderEntry) entry).getPlaceholder()));
        } else if (entry instanceof TankerTemplateEntry.ConditionalEntry) {
            TankerTemplateEntry.ConditionalEntry conditionalEntry = (TankerTemplateEntry.ConditionalEntry) entry;
            processCondition(conditionalEntry, phs, e -> addEntry(textEntries, phs, e));
        } else if (entry instanceof TankerTemplateEntry.PluralEntry) {
            TankerTemplateEntry.PluralEntry pluralEntry = (TankerTemplateEntry.PluralEntry) entry;
            processPlural(pluralEntry, phs, e -> addEntry(textEntries, phs, e));
        }
    }

    private static <T> void processPlural(TankerTemplateEntry.PluralEntry entry, TankerPlaceholders<T> phs, Consumer<TankerTemplateEntry> onAppend) {
        long count = phs.getNumber(entry.getCountParam());
        List<TankerTemplateEntry> entries;
        if (count == 0) {
            entries = entry.getNoneEntries();
        } else {
            int lastNumber = (int) (count % 10);
            int last2Numbers = (int) (count % 100);
            if (lastNumber == 1 && last2Numbers != 11) {
                entries = entry.getOneEntries();
            } else {
                if (lastNumber > 1 && lastNumber < 5 && (last2Numbers < 10 || last2Numbers > 20)) {
                    entries = entry.getSomeEntries();
                } else {
                    entries = entry.getManyEntries();
                }
            }
        }
        entries.forEach(onAppend);
    }

    private static <T> void processCondition(TankerTemplateEntry.ConditionalEntry entry, TankerPlaceholders<T> phs, Consumer<TankerTemplateEntry> onAppend) {
        Object condition = phs.get(entry.getCondition());
        if (condition == null || Boolean.FALSE.toString().equals(condition)) {
            entry.getFalseEntries().forEach(onAppend);
        } else {
            entry.getTrueEntries().forEach(onAppend);
        }
    }

    public static class StaticBuilder {
        private final Map<LanguageEnum, String> texts = new EnumMap<>(LanguageEnum.class);

        public StaticBuilder add(LanguageEnum lang, String text) {
            texts.put(lang, text);
            return this;
        }

        public <Builder> TankerStaticKeyWithBuilder<Builder> build(Function<List<TankerTemplateEntry>, Builder> builderFactory) {
            return new TankerStaticKeyWithBuilder<>(texts, builderFactory);
        }

        public TankerStaticKey build() {
            return new TankerStaticKey(texts);
        }
    }
}

