package ru.yandex.webmaster3.viewer.http.achievements;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.concurrency.AsyncCtx;
import ru.yandex.webmaster3.core.concurrency.AsyncTask;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.iks.data.Sqi;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.searchquery.OrderDirection;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.achievements.dao.AchievementsCHDao;
import ru.yandex.webmaster3.storage.achievements.dao.AchievementsData;
import ru.yandex.webmaster3.storage.achievements.dao.HostAchievementsRivalsYDao;
import ru.yandex.webmaster3.storage.achievements.model.AchievementInfo;
import ru.yandex.webmaster3.storage.achievements.model.AchievementProblem;
import ru.yandex.webmaster3.storage.achievements.model.AchievementService;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.storage.services.SiteServiceService;
import ru.yandex.webmaster3.storage.turbo.service.TurboSearchUrlsStatisticsService;
import ru.yandex.webmaster3.storage.ugc.SiteRating;
import ru.yandex.webmaster3.storage.ugc.SiteRatingService;
import ru.yandex.webmaster3.viewer.http.concurrency.AsyncAction;

/**
 * Created by Oleg Bazdyrev on 05/10/2018.
 */
@Slf4j
@ReadAction
@Category("achievements")
@AllArgsConstructor(onConstructor_ = @Autowired)
@Description("Получение ачивок для хоста и его конкурентов")
public class GetHostAndRivalsAchievementsAction extends AsyncAction<GetHostAndRivalsAchievementsRequest,
        GetHostAndRivalsAchievementsResponse> {
    private final AchievementsCHDao mdbAchievementsCHDao;
    private final SiteServiceService siteServiceService;
    private final SiteRatingService siteRatingService;
    private final HostAchievementsRivalsYDao hostAchievementsRivalsYDao;
    private final TurboSearchUrlsStatisticsService turboSearchUrlsStatisticsService;
    private final IksService iksService;
    private final AchievementService achievementService;
    private final HostOwnerService hostOwnerService;
    private final FeedsService feedsService;

    public GetHostAndRivalsAchievementsResponse processAsync(AsyncCtx ctx,
                                                             GetHostAndRivalsAchievementsRequest request) {
        List<String> rivals = new ArrayList<>(hostAchievementsRivalsYDao.getRivalsFiltered(request.getHostId()));
        String requestHost = request.getHostId().getPunycodeHostname();
        AchievementTld tld = request.getTld();
        List<String> allHosts = new ArrayList<>(rivals);
        allHosts.add(requestHost);

        Map<String, String> host2Owner = allHosts.stream()
                .distinct()
                .collect(Collectors.toMap(Function.identity(), hostOwnerService::getMascotHostOwner));
        Set<String> owners = new HashSet<>(host2Owner.values());

        // получаем иксы
        AsyncTask<Map<String, Sqi>> sqiTask = ctx.fork(() -> iksService.getSqiForOwners(ctx, owners));

        // получаем ачивки
        List<AsyncTask<Pair<String, SiteRating>>> srvTask = rivals.stream()
                .distinct()
                .map(owner -> ctx.fork(() -> Pair.of(owner, siteRatingService.getRating(host2Owner.get(owner)))))
                .toList();


        // получаем ачивки
        AsyncTask<Map<String, AchievementsData>> achieveTask =
                ctx.fork(() -> mdbAchievementsCHDao.getBatchAchievementsAndProblems(tld, host2Owner.values()));

        // дождемся выполнения тасок
        Map<String, Sqi> owner2Sqi = sqiTask.join();

        Map<String, AchievementsData> owner2Ach = achieveTask.join();

        Map<String, SiteRating> owner2Rating = srvTask.stream()
                .map(AsyncTask::join)
                .filter(a -> a.getRight() != null)
                .collect(Collectors.toMap(Pair::getLeft, Pair::getRight));

        // соберем все ачивки в мапу
        List<AchievementProblem> hostProblems = Collections.emptyList();
        Map<String, Map<AchievementType, AchievementInfo>> map = new HashMap<>();
        for (var entry : host2Owner.entrySet()) {
            String host = entry.getKey();
            String owner = entry.getValue();
            AchievementsData pair = owner2Ach.get(owner);
            if (requestHost.equals(host)) {
                hostProblems = pair.getProblems();
            }

            List<AchievementInfo> achievements = new ArrayList<>(pair.getInfos());
            if (request.getTld() == AchievementTld.RU) {
                Sqi sqi = owner2Sqi.get(owner);
                final int iks = sqi == null ? 0 : sqi.getIks();
                achievements.add(new AchievementInfo.SqiAchievementInfo(null, null, iks));
            }
            Map<AchievementType, AchievementInfo> achievementsMap = achievements.stream()
                    .collect(Collectors.toMap(AchievementInfo::getType, Function.identity()));
            map.put(host, achievementsMap);
        }

        // формируем список для таблички
        List<RivalAchievements> rows = new ArrayList<>();

        var siteServiceInfoByHost = siteServiceService.getBatchRivalInfo(rivals, requestHost);
        List<String> domains = rivals.stream()
                .map(WwwUtil::cutWWWAndM)
                .toList();
        Set<String> domainsWithGoodFeedsInSearch = feedsService.getDomainsWithTvInSearch(domains);
        for (String host : rivals) {
            var siteServiceInfo = siteServiceInfoByHost.get(host);
            String domain = WwwUtil.cutWWWAndM(host);
            if (requestHost.equals(host)) {
                rows.add(new RivalAchievements(
                        true,
                        host,
                        map.getOrDefault(host, Collections.emptyMap()),
                        turboSearchUrlsStatisticsService.countTurboPages(domain),
                        siteServiceInfo,
                        owner2Rating.get(host),
                        domainsWithGoodFeedsInSearch.contains(domain)
                ));
            } else {
                rows.add(new RivalAchievements(
                        false,
                        host,
                        map.getOrDefault(host, Collections.emptyMap()),
                        null,
                        siteServiceInfo,
                        owner2Rating.get(host),
                        domainsWithGoodFeedsInSearch.contains(domain)
                ));
            }
        }

        // отсортируем
        if (request.getOrderBy() != null) {
            Comparator<RivalAchievements> comparator = request.getOrderBy().getComparator();
            if (request.getOrderDirection() == OrderDirection.DESC) {
                comparator = comparator.reversed();
            }
            rows.sort(comparator);
        }

        // ачивки искомого хоста
        Map<AchievementType, AchievementInfo> hostAchievements = map.getOrDefault(requestHost, Collections.emptyMap());

        return new GetHostAndRivalsAchievementsResponse(hostAchievements, hostProblems, rows,
                request.getTld().isCanHaveTld(), achievementService.getAllAchievementTypes(request.getTld()));
    }
}
