package ru.yandex.webmaster3.viewer.http.achievements;

import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.achievements.dao.AchievementsCHDao;
import ru.yandex.webmaster3.storage.achievements.model.AchievementInfo;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;

/**
 * Created by shabashoff.
 */
@ReadAction
@Category("achievements")
@AllArgsConstructor(onConstructor_ = @Autowired)
@Description("Получение грейда скорости")
public class GetSpeedGradeAction extends Action<GetSpeedGradeRequest, GetSpeedGradeResponse> {
    private static final AchievementTld DEFAULT_TLD = AchievementTld.RU;
    private final AchievementsCHDao mdbAchievementsCHDao;
    private final HostOwnerService hostOwnerService;
    private final AbtService abtService;

    @Override
    public GetSpeedGradeResponse process(GetSpeedGradeRequest request) throws WebmasterException {
        @NotNull WebmasterHostId notNullOwner =
                Optional.ofNullable(hostOwnerService.getHostOwner(request.getHostId())).orElse(request.getHostId());
        @NotNull String domain = WwwUtil.cutWWWAndM(notNullOwner.getPunycodeHostname());
        boolean turbo = false;
        int speedGrade = 0;

        Map<AchievementType, AchievementInfo> map = mdbAchievementsCHDao
                .getAll(DEFAULT_TLD, domain)
                .getInfos()
                .stream()
                .collect(W3Collectors.toEnumMap(AchievementInfo::getType, Function.identity(), AchievementType.class));

        if (map.containsKey(AchievementType.TURBO)) {
            var info = map.get(AchievementType.TURBO);
            if (info instanceof AchievementInfo.TurboAchievementInfo) {
                turbo = ((AchievementInfo.TurboAchievementInfo) info).isValue();
            }
        }
        if (map.containsKey(AchievementType.MOBILE_SPEED)) {
            var info = map.get(AchievementType.MOBILE_SPEED);
            if (info instanceof AchievementInfo.MobileSpeedAchievementInfo) {
                speedGrade = ((AchievementInfo.MobileSpeedAchievementInfo) info).getValue();
            }
        }
        if (abtService.isInExperiment(request.getUserId(), Experiment.SPEED_GRADE_ACHIEVEMENT)) {
            return new GetSpeedGradeResponse(domain, speedGrade, turbo);
        }

        return new GetSpeedGradeResponse(domain, 0, turbo);
    }
}
