package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.util.List;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.favicon.FaviconInfo;
import ru.yandex.webmaster3.storage.favicon.HostFaviconsCHDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.checklist.extras.FaviconErrorsSamplesAction.FaviconErrorsSamplesResponse;
import ru.yandex.webmaster3.core.util.http.YandexMimeType;

/**
 * @author avhaliullin
 */
@ReadAction
@Description("Получить примеры проблемных фавиконок")
@Category("checklist")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/checklist/extra/faviconErrorSamples")
public class FaviconErrorsSamplesAction extends AbstractUserVerifiedHostAction<PageSamplesRequest,
        FaviconErrorsSamplesResponse> {

    private final HostFaviconsCHDao mdbHostFaviconsCHDao;

    @Override
    public FaviconErrorsSamplesResponse process(PageSamplesRequest request) {
        List<FaviconInfo> badFavicons = mdbHostFaviconsCHDao.getFaviconsSamples(request.getHostId(),
                SiteProblemTypeEnum.FAVICON_ERROR);
        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        List<FaviconErrorSample> samples = badFavicons.stream()
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .map(FaviconErrorSample::fromFaviconInfo)
                .collect(Collectors.toList());

        return new FaviconErrorsSamplesResponse(samples, badFavicons.size());
    }

    @Getter
    @RequiredArgsConstructor
    public static class FaviconErrorsSamplesResponse implements ActionResponse.NormalResponse {
        private final List<FaviconErrorSample> samples;
        private final int totalSamples;
    }

    public enum FaviconErrorType {
        NO_INDEX,
        REDIRECT,
        BAD_HTTP_CODE,
        BAD_SIZE,
        BAD_MIME_TYPE,
    }

    @Getter
    @RequiredArgsConstructor
    public static final class FaviconErrorSample {
        private final String url;
        private final HttpCodeInfo httpCode;
        private final YandexMimeType mimeType;
        private final FaviconErrorType errorType;

        public static FaviconErrorSample fromFaviconInfo(FaviconInfo sample) {
            FaviconErrorType errorType;
            if (sample.getHttpCode() >= 300 && sample.getHttpCode() < 400) {
                errorType = FaviconErrorType.REDIRECT;
            } else if (sample.getHttpCode() >= 400) {
                errorType = FaviconErrorType.BAD_HTTP_CODE;
            } else {
                errorType = FaviconErrorType.BAD_MIME_TYPE;
            }
            return new FaviconErrorSample(sample.getUrl(), new HttpCodeInfo(sample.getHttpCode()),
                    YandexMimeType.R.fromValueOrUnknown(sample.getMimeType()), errorType);
        }
    }
}
