package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.hoststat.HostStatistics;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.request.PagerAware;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.PageUtils;
import ru.yandex.webmaster3.storage.hoststat.HostStatisticsCHDao;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@ReadAction
@Category("checklist")
@Component("/checklist/extras/slowResponsePageSamples")
public class SlowResponseWithExamplesPageSampleAction extends AbstractUserVerifiedHostAction<SlowResponseWithExamplesPageSampleAction.Request, SlowResponseWithExamplesPageSampleAction.Response> {

    private final HostStatisticsCHDao hostStatisticsCHDao;

    @Override
    public Response process(Request request) {
        HostStatistics hostStatistics = hostStatisticsCHDao.getHostStatisticsWithSamples(request.getHostId());
        if (hostStatistics == null) {
            return new Response(Collections.emptyList(), 0);
        }
        PageUtils.Pager pager = PageUtils.getPage(request.getP(), request.getPSize());
        List<Response.PageSample> results = hostStatistics.getSlowPageSamples().stream()
                .skip(pager.toRangeStart())
                .limit(pager.getPageSize())
                .map(pageSample -> Response.PageSample.create(request.getHostId(), pageSample))
                .collect(Collectors.toList());

        return new Response(results, hostStatistics.getSlowPageSamples().size());
    }

    @Data
    public static class Request extends AbstractUserVerifiedHostRequest implements PagerAware {
        private int pSize;
        private int p;
    }

    @Value
    public static class Response implements ActionResponse.NormalResponse {
        List<PageSample> samples;
        int totalSamples;

        @Value
        public static class PageSample {
            URL url;
            Long lastAccess;
            Long fetchTime;

            public static PageSample create(WebmasterHostId hostId, HostStatistics.PageSample sample) {
                try {
                    return new PageSample(new URL(IdUtils.hostIdToUrl(hostId) + sample.getPath()), sample.getLastAccess(), sample.getFetchTime());
                } catch (MalformedURLException e) {
                    throw new RuntimeException("Malformed url in page samples", e);
                }
            }
        }
    }
}
