package ru.yandex.webmaster3.viewer.http.checklist.extras;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.utils.URIBuilder;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.AbstractProblemInfo;
import ru.yandex.webmaster3.storage.checklist.data.ExtendedProblem;
import ru.yandex.webmaster3.storage.checklist.data.TurboHostBanSample;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostAction;
import ru.yandex.webmaster3.viewer.http.AbstractUserVerifiedHostRequest;

/**
 * @author: ishalaru
 * DATE: 15.07.2019
 */
@Slf4j
@ReadAction
@Description("Получить примеры проблемных страниц с банами хостов турбо ")
@Category("checklist")
@Component("/checklist/extras/turboHostBanExtendedSamples/v2")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboHostBanExtendedSamplesV2Action extends AbstractUserVerifiedHostAction<TurboHostBanExtendedSamplesV2Action.TurboBanSamplesRequest,
        TurboHostBanExtendedSamplesV2Action.TurboBanSamplesResponse> {
    private static final ObjectMapper OM = new ObjectMapper().configure(JsonGenerator.Feature.ESCAPE_NON_ASCII, true);
    private final SiteProblemsService siteProblemsService;
    private final ChecklistPageSamplesService checklistPageSamplesService;
    public static final int COUNT_SAMPLES_BY_REASON = 3;

    @Override
    public TurboBanSamplesResponse process(TurboBanSamplesRequest request) {
        SiteProblemTypeEnum problemType = SiteProblemTypeEnum.TURBO_HOST_BAN;
        AbstractProblemInfo problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), problemType);

        if (problemInfo == null || problemInfo.getState() == problemType.getDefaultState()) {
            problemType = SiteProblemTypeEnum.TURBO_HOST_BAN_INFO;
            problemInfo = siteProblemsService.getRealTimeProblemInfo(request.getHostId(), problemType);
            if (problemInfo == null || problemInfo.getState() == problemType.getDefaultState()) {
                return new TurboBanSamplesResponse(Collections.emptyList(), DateTime.now(), false);
            }
        }

        final ChecklistSamplesType checklistSamplesType = ChecklistSamplesType.byProblemType(problemType);
        List<String> workValues = checklistPageSamplesService.getSamples(request.getHostId(), checklistSamplesType);

        List<TurboHostBanSample> samples = new ArrayList<>(workValues.size());
        for (String item : workValues) {
            TurboHostBanSample value = null;
            try {
                value = OM.readValue(item, TurboHostBanSample.class);
            } catch (IOException e) {
                log.error("Error while parsing turbo host ban sample {}", item, e);
            }
            if (value != null) {
                samples.add(value);
            }
        }
        final String domain = WwwUtil.cutWWWAndM(request.getHostId());
        List<TurboHostBanSample> samplesOnlyDomain = samples.stream()
                .filter(sample -> isDomainSamples(domain, sample))
                .collect(Collectors.toList());
        // если есть примеры только для данного домена, показываем их. Иначе выбираем примеры из всех(так как проблема по овнеру)
        if (!samplesOnlyDomain.isEmpty()) {
            samples = samplesOnlyDomain;
        }

        List<TurboBanSampleByReason> result = samples.stream()
                .flatMap(x -> TurboBanSample.fromHostBanSample(x).stream())
                .collect(Collectors.groupingBy(TurboBanSample::getReason))
                .entrySet().stream()
                .map(entry -> new TurboBanSampleByReason(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
        ExtendedProblem<AbstractProblemInfo> extendedProblem =
                siteProblemsService.getExtendedProblems(request.getHostId(), List.of(problemInfo)).get(0);

        return new TurboBanSamplesResponse(result, problemInfo.getLastUpdate(), extendedProblem.isRecheckRequested());
    }

    private static boolean isDomainSamples(String domain, TurboHostBanSample sample) {
        try {
            String originalUrl = sample.getOriginalUrl();
            if (originalUrl == null) {
                return false;
            }
            URIBuilder origUriBuilder = new URIBuilder(originalUrl);
            if (origUriBuilder.getHost() == null) {
                return true;
            }
            String origUriDomain = WwwUtil.cutWWWAndM(origUriBuilder.getHost());
            return origUriDomain.equals(domain);
        } catch (URISyntaxException ignore) {

        }
        return false;
    }

    @AllArgsConstructor
    @Getter
    public static class TurboBanSamplesResponse implements ActionResponse.NormalResponse {
        private final List<TurboBanSampleByReason> samples;
        private final DateTime lastCheck;
        private final boolean recheckRequested;
    }

    public static class TurboBanSamplesRequest extends AbstractUserVerifiedHostRequest {

    }


    @AllArgsConstructor
    @Getter
    public static class TurboBanSample {
        private final String originalUrl;
        private final String prodTurboUrl;
        private final String reason;

        public static List<TurboBanSample> fromHostBanSample(TurboHostBanSample sample) {
            return sample.getReasons().stream().map(reason -> new TurboBanSample(sample.getOriginalUrl(),
                    sample.getFrozenUrl(), reason))
                    .collect(Collectors.toList());
        }
    }

    @Getter
    public static class TurboBanSampleByReason {
        private final String reason;
        private final List<TurboBanSample> samples;

        //сортируем по названию чтобы каждый раз отдавать одинаковые
        public TurboBanSampleByReason(String reason, List<TurboBanSample> samples) {
            this.reason = reason;
            this.samples = samples.stream()
                    .sorted(Comparator.comparing(TurboBanSample::getOriginalUrl).reversed())
                    .limit(COUNT_SAMPLES_BY_REASON)
                    .collect(Collectors.toList());
        }
    }
}
